<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\ScheduledPost;
use App\Models\SocialAccount;
use App\Models\Campaign;
use App\Models\AiJob;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Http\Requests\StoreScheduledPostRequest;
use App\Http\Requests\UpdateScheduledPostRequest;

class ScheduledPostController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = ScheduledPost::where('brand_id', auth()->user()->brand_id)
            ->with(['socialAccount', 'campaign', 'createdBy']);

        // Apply filters
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('campaign_id')) {
            $query->where('campaign_id', $request->campaign_id);
        }

        // Get scheduled posts
        $scheduledPosts = $query->latest()->paginate(20);

        // Get campaigns for filter dropdown
        $campaigns = Campaign::where('brand_id', auth()->user()->brand_id)->get();

        return view('brand.scheduled-posts.index', compact('scheduledPosts', 'campaigns'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get social accounts for the brand
        $socialAccounts = SocialAccount::where('brand_id', auth()->user()->brand_id)
            ->where('tenant_id', tenant('id'))
            ->get();

        // Get campaigns for the brand
        $campaigns = Campaign::where('brand_id', auth()->user()->brand_id)->get();

        return view('brand.scheduled-posts.create', compact('socialAccounts', 'campaigns'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreScheduledPostRequest $request)
    {

        try {
            return DB::transaction(function () use ($request) {
                // Create the scheduled post
                $scheduledPost = ScheduledPost::create([
                    'brand_id' => auth()->user()->brand_id,
                    'social_account_id' => $request->social_account_id,
                    'caption' => $request->caption,
                    'media_refs' => $request->media_refs,
                    'publish_time' => $request->publish_time,
                    'timezone' => $request->timezone,
                    'status' => $request->approval_required ? 'pending_approval' : 'scheduled',
                    'retry_policy' => [
                        'max_attempts' => 3,
                        'backoff' => 'exponential',
                        'base_delay' => 60,
                        'max_delay' => 86400,
                        'multiplier' => 2
                    ],
                    'idempotency_key' => Str::uuid(),
                    'campaign_id' => $request->campaign_id,
                    'post_visibility' => $request->post_visibility,
                    'recurrence' => $request->recurrence,
                    'approval_required' => $request->approval_required ?? false,
                    'created_by' => auth()->id(),
                ]);

                // Send notification if approval is required
                if ($request->approval_required) {
                    // TODO: Send notification to approvers
                }

                return redirect()->route('brand.scheduled-posts.show', $scheduledPost)
                    ->with('success', 'Post scheduled successfully.');
            });
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to schedule post: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(ScheduledPost $scheduledPost)
    {
        // Ensure the user can only view their own scheduled posts
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        $scheduledPost->load(['socialAccount', 'campaign', 'createdBy', 'approvedBy', 'postAttempts']);

        return view('brand.scheduled-posts.show', compact('scheduledPost'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ScheduledPost $scheduledPost)
    {
        // Ensure the user can only edit their own scheduled posts
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        // Get social accounts for the brand
        $socialAccounts = SocialAccount::where('brand_id', auth()->user()->brand_id)
            ->where('tenant_id', tenant('id'))
            ->get();

        // Get campaigns for the brand
        $campaigns = Campaign::where('brand_id', auth()->user()->brand_id)->get();

        $scheduledPost->load(['socialAccount', 'campaign']);

        return view('brand.scheduled-posts.edit', compact('scheduledPost', 'socialAccounts', 'campaigns'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateScheduledPostRequest $request, ScheduledPost $scheduledPost)
    {
        // Ensure the user can only update their own scheduled posts
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        try {
            $scheduledPost->update([
                'social_account_id' => $request->social_account_id,
                'caption' => $request->caption,
                'media_refs' => $request->media_refs,
                'publish_time' => $request->publish_time,
                'timezone' => $request->timezone,
                'campaign_id' => $request->campaign_id,
                'post_visibility' => $request->post_visibility,
                'recurrence' => $request->recurrence,
            ]);

            return redirect()->route('brand.scheduled-posts.show', $scheduledPost)
                ->with('success', 'Post updated successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to update post: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ScheduledPost $scheduledPost)
    {
        // Ensure the user can only delete their own scheduled posts
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        try {
            $scheduledPost->delete();

            return redirect()->route('brand.scheduled-posts.index')
                ->with('success', 'Post deleted successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to delete post: ' . $e->getMessage());
        }
    }

    /**
     * Publish a scheduled post immediately.
     */
    public function publishNow(ScheduledPost $scheduledPost)
    {
        // Ensure the user can only publish their own scheduled posts
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        // Check if post is in a valid status for immediate publishing
        if (!in_array($scheduledPost->status, ['scheduled', 'draft', 'pending_approval'])) {
            return back()->with('error', 'Post cannot be published in its current status.');
        }

        try {
            // Dispatch job to process the scheduled post immediately
            \App\Jobs\ProcessScheduledPost::dispatch($scheduledPost);
            
            // Mark as publishing
            $scheduledPost->markPublishing();

            return back()->with('success', 'Post is being published now.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to publish post: ' . $e->getMessage());
        }
    }

    /**
     * Reschedule a post.
     */
    public function reschedule(ScheduledPost $scheduledPost, Request $request)
    {
        // Ensure the user can only reschedule their own scheduled posts
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        $request->validate([
            'publish_time' => 'required|date',
            'timezone' => 'required|string',
        ]);

        try {
            $scheduledPost->update([
                'publish_time' => $request->publish_time,
                'timezone' => $request->timezone,
                'status' => 'scheduled',
            ]);

            return back()->with('success', 'Post has been rescheduled.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to reschedule post: ' . $e->getMessage());
        }
    }

    /**
     * Cancel a scheduled post.
     */
    public function cancel(ScheduledPost $scheduledPost)
    {
        // Ensure the user can only cancel their own scheduled posts
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        try {
            $scheduledPost->markCancelled();

            return back()->with('success', 'Post has been cancelled.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to cancel post: ' . $e->getMessage());
        }
    }

    /**
     * Approve a scheduled post.
     */
    public function approve(ScheduledPost $scheduledPost)
    {
        // Ensure the user can approve scheduled posts for their brand
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        // Check if approval is required
        if (!$scheduledPost->approval_required) {
            return back()->with('error', 'Approval is not required for this post.');
        }

        try {
            $scheduledPost->approve(auth()->user());

            return back()->with('success', 'Post has been approved and scheduled.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to approve post: ' . $e->getMessage());
        }
    }

    /**
     * Deny a scheduled post.
     */
    public function deny(ScheduledPost $scheduledPost)
    {
        // Ensure the user can deny scheduled posts for their brand
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        // Check if approval is required
        if (!$scheduledPost->approval_required) {
            return back()->with('error', 'Approval is not required for this post.');
        }

        try {
            $scheduledPost->deny();

            return back()->with('success', 'Post has been denied.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to deny post: ' . $e->getMessage());
        }
    }

    /**
     * Retry processing a failed post.
     */
    public function retry(ScheduledPost $scheduledPost)
    {
        // Ensure the user can only retry their own scheduled posts
        if ($scheduledPost->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        if ($scheduledPost->status != 'failed') {
            return back()->with('error', 'Only failed posts can be retried.');
        }

        try {
            // Reset attempt count and error message
            $scheduledPost->update([
                'status' => 'scheduled',
                'error_message' => null,
                'attempt_count' => 0,
                'last_attempt_at' => null,
                'next_retry_at' => null,
            ]);

            return back()->with('success', 'Post processing has been restarted.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to retry post processing: ' . $e->getMessage());
        }
    }
}
