<?php

namespace App\Http\Controllers\Creator;

use App\Http\Controllers\Controller;
use App\Models\Campaign;
use App\Models\CampaignMilestone;
use App\Models\CampaignBudgetAllocation;
use App\Jobs\EnqueueAiTaskJob;
use Illuminate\Http\Request;

class CampaignPlannerController extends Controller
{
    /**
     * Show the campaign planner view
     *
     * @param  \App\Models\Campaign  $campaign
     * @return \Illuminate\Http\Response
     */
    public function show(Campaign $campaign)
    {
        // Ensure the creator is part of this campaign
        if (!$campaign->creators()->where('creator_id', auth()->id())->exists()) {
            abort(403);
        }
        
        // Load related data
        $campaign->load(['milestones', 'budgetAllocations']);
        
        return view('creator.campaigns.planner', compact('campaign'));
    }
    
    /**
     * Store a new milestone for the campaign
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Campaign  $campaign
     * @return \Illuminate\Http\Response
     */
    public function storeMilestone(Request $request, Campaign $campaign)
    {
        // Ensure the creator is part of this campaign
        if (!$campaign->creators()->where('creator_id', auth()->id())->exists()) {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'due_date' => 'nullable|date',
            'status' => 'required|in:pending,in_progress,completed',
        ]);
        
        $milestone = new CampaignMilestone([
            'campaign_id' => $campaign->id,
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'due_date' => $validated['due_date'] ?? null,
            'status' => $validated['status'],
            // For creator campaigns, the creator is automatically assigned to their own milestones
            'assigned_creator_id' => auth()->id(),
        ]);
        
        $milestone->save();
        
        return response()->json(['success' => true, 'milestone' => $milestone]);
    }
    
    /**
     * Update an existing milestone
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\CampaignMilestone  $milestone
     * @return \Illuminate\Http\Response
     */
    public function updateMilestone(Request $request, CampaignMilestone $milestone)
    {
        // Ensure the creator is part of this campaign
        if (!$milestone->campaign->creators()->where('creator_id', auth()->id())->exists()) {
            abort(403);
        }
        
        // Ensure the milestone is assigned to this creator
        if ($milestone->assigned_creator_id !== auth()->id()) {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'due_date' => 'nullable|date',
            'status' => 'required|in:pending,in_progress,completed',
        ]);
        
        $milestone->update($validated);
        
        return response()->json(['success' => true, 'milestone' => $milestone]);
    }
    
    /**
     * Delete a milestone
     *
     * @param  \App\Models\CampaignMilestone  $milestone
     * @return \Illuminate\Http\Response
     */
    public function deleteMilestone(CampaignMilestone $milestone)
    {
        // Ensure the creator is part of this campaign
        if (!$milestone->campaign->creators()->where('creator_id', auth()->id())->exists()) {
            abort(403);
        }
        
        // Ensure the milestone is assigned to this creator
        if ($milestone->assigned_creator_id !== auth()->id()) {
            abort(403);
        }
        
        $milestone->delete();
        
        return response()->json(['success' => true]);
    }
    
    /**
     * Request AI planning suggestions for the campaign
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Campaign  $campaign
     * @return \Illuminate\Http\Response
     */
    public function requestAiPlanning(Request $request, Campaign $campaign)
    {
        // Ensure the creator is part of this campaign
        if (!$campaign->creators()->where('creator_id', auth()->id())->exists()) {
            abort(403);
        }
        
        // For creator campaigns, we'll create a simplified AI planning request
        // that focuses on content creation and posting schedule
        
        // Prepare the payload for the AI task
        $payload = [
            'campaign_id' => $campaign->id,
            'name' => $campaign->name,
            'description' => $campaign->description,
            'objective' => $campaign->objective,
            'deliverables' => $campaign->deliverables,
            'audience' => $campaign->audience,
            'start_date' => $campaign->start_date,
            'end_date' => $campaign->end_date,
            'creator_id' => auth()->id(),
        ];
        
        // Dispatch the AI task job
        EnqueueAiTaskJob::dispatch('creator_campaign_planning', null, auth()->id(), $payload);
        
        // For demonstration purposes, let's also simulate the AI response
        // In a real implementation, this would be handled by an external AI processing system
        $aiTask = \App\Models\AiTask::where('type', 'creator_campaign_planning')
            ->where('creator_id', auth()->id())
            ->orderBy('created_at', 'desc')
            ->first();
            
        if ($aiTask) {
            // Simulate AI processing completion
            $aiTask->update([
                'status' => 'completed',
                'result_text' => 'AI planning suggestions generated successfully',
                'result_meta' => [
                    'campaign_id' => $campaign->id,
                    'milestones' => [
                        [
                            'name' => 'Content Planning',
                            'description' => 'Plan content themes and ideas',
                            'due_date' => now()->addDays(3)->format('Y-m-d'),
                            'status' => 'pending',
                        ],
                        [
                            'name' => 'Content Creation',
                            'description' => 'Create content for all deliverables',
                            'due_date' => now()->addDays(10)->format('Y-m-d'),
                            'status' => 'pending',
                        ],
                        [
                            'name' => 'Content Review',
                            'description' => 'Review and finalize all content',
                            'due_date' => now()->addDays(14)->format('Y-m-d'),
                            'status' => 'pending',
                        ],
                        [
                            'name' => 'Content Posting',
                            'description' => 'Post content on all platforms',
                            'due_date' => now()->addDays(17)->format('Y-m-d'),
                            'status' => 'pending',
                        ],
                        [
                            'name' => 'Performance Analysis',
                            'description' => 'Analyze content performance and prepare report',
                            'due_date' => now()->addDays(21)->format('Y-m-d'),
                            'status' => 'pending',
                        ],
                    ],
                    'posting_schedule' => [
                        [
                            'date' => now()->addDays(5)->format('Y-m-d'),
                            'platform' => 'Instagram',
                            'content_type' => 'Reel',
                            'theme' => 'Behind the scenes',
                        ],
                        [
                            'date' => now()->addDays(7)->format('Y-m-d'),
                            'platform' => 'Instagram',
                            'content_type' => 'Story',
                            'theme' => 'Product showcase',
                        ],
                        [
                            'date' => now()->addDays(12)->format('Y-m-d'),
                            'platform' => 'Instagram',
                            'content_type' => 'Post',
                            'theme' => 'Campaign highlight',
                        ],
                        [
                            'date' => now()->addDays(15)->format('Y-m-d'),
                            'platform' => 'TikTok',
                            'content_type' => 'Video',
                            'theme' => 'Challenge participation',
                        ],
                    ],
                ],
            ]);
            
            // Dispatch the job to process the AI planning results
            \App\Jobs\ProcessCampaignPlanningAiJob::dispatch($aiTask);
        }
        
        return response()->json(['success' => true, 'message' => 'AI planning request submitted successfully']);
    }
    
    /**
     * Export campaign as summary
     *
     * @param  \App\Models\Campaign  $campaign
     * @return \Illuminate\Http\Response
     */
    public function exportCampaign(Campaign $campaign)
    {
        // Ensure the creator is part of this campaign
        if (!$campaign->creators()->where('creator_id', auth()->id())->exists()) {
            abort(403);
        }
        
        // Get campaign data for export
        $campaign->load(['milestones', 'budgetAllocations']);
        
        // Prepare export data
        $exportData = [
            'campaign_name' => $campaign->name,
            'description' => $campaign->description,
            'objective' => $campaign->objective,
            'deliverables' => $campaign->deliverables,
            'audience' => $campaign->audience,
            'start_date' => $campaign->start_date ? $campaign->start_date->format('Y-m-d') : null,
            'end_date' => $campaign->end_date ? $campaign->end_date->format('Y-m-d') : null,
            'status' => $campaign->status,
            'milestones' => $campaign->milestones->map(function ($milestone) {
                return [
                    'name' => $milestone->name,
                    'description' => $milestone->description,
                    'due_date' => $milestone->due_date ? $milestone->due_date->format('Y-m-d') : null,
                    'status' => $milestone->status,
                ];
            }),
            'budget' => $campaign->budget,
            'currency' => $campaign->currency,
            'exported_at' => now()->format('Y-m-d H:i:s'),
            'exported_by' => auth()->user()->name,
        ];
        
        // For now, we'll just return a JSON response
        // In a real implementation, this would generate a PDF or document
        return response()->json([
            'success' => true,
            'message' => 'Campaign exported successfully',
            'data' => $exportData
        ]);
    }
    
    /**
     * Create a new campaign for the creator
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function createCampaign(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'objective' => 'nullable|string',
            'deliverables' => 'nullable|string',
            'audience' => 'nullable|string',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'budget' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|max:3',
        ]);
        
        // Create the campaign with the creator as the owner
        $campaign = new Campaign([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'objective' => $validated['objective'] ?? null,
            'deliverables' => $validated['deliverables'] ?? null,
            'audience' => $validated['audience'] ?? null,
            'budget' => $validated['budget'] ?? null,
            'currency' => $validated['currency'] ?? 'USD',
            'start_date' => $validated['start_date'] ?? null,
            'end_date' => $validated['end_date'] ?? null,
            'status' => 'draft',
            'created_by' => auth()->id(),
        ]);
        
        $campaign->save();
        
        // Attach the creator to the campaign
        $campaign->creators()->attach(auth()->id(), [
            'status' => 'owner',
            'invitation_sent_at' => now(),
            'responded_at' => now(),
        ]);
        
        return response()->json(['success' => true, 'campaign' => $campaign]);
    }
}