<?php

namespace App\Http\Controllers\Creator;

use App\Http\Controllers\Controller;
use App\Models\Content;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ContentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $contents = Content::where('creator_id', Auth::id())->get();
        return view('creator.content.index', compact('contents'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('creator.content.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'status' => 'required|in:draft,pending,published',
        ]);

        $content = new Content();
        $content->brand_id = Auth::user()->brand_id;
        $content->creator_id = Auth::id();
        $content->title = $request->title;
        $content->body = $request->body;
        $content->status = $request->status;
        $content->save();

        return redirect()->route('creator.content.index')->with('success', 'Content created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Content $content)
    {
        // Ensure creator can only view their own content
        if ($content->creator_id !== Auth::id()) {
            abort(403);
        }
        
        return view('creator.content.show', compact('content'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Content $content)
    {
        // Ensure creator can only edit their own content
        if ($content->creator_id !== Auth::id()) {
            abort(403);
        }
        
        return view('creator.content.edit', compact('content'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Content $content)
    {
        // Ensure creator can only update their own content
        if ($content->creator_id !== Auth::id()) {
            abort(403);
        }
        
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'status' => 'required|in:draft,pending,published',
        ]);

        $content->title = $request->title;
        $content->body = $request->body;
        $content->status = $request->status;
        $content->save();

        return redirect()->route('creator.content.index')->with('success', 'Content updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Content $content)
    {
        // Ensure creator can only delete their own content
        if ($content->creator_id !== Auth::id()) {
            abort(403);
        }
        
        $content->delete();
        
        return redirect()->route('creator.content.index')->with('success', 'Content deleted successfully.');
    }
}