<?php

namespace App\Http\Controllers;

use App\Models\Plan;
use App\Models\Subscription;
use App\Services\RazorpayService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SubscriptionController extends Controller
{
    protected $razorpayService;

    public function __construct(RazorpayService $razorpayService)
    {
        $this->razorpayService = $razorpayService;
    }

    /**
     * Display pricing plans.
     */
    public function index()
    {
        $plans = Plan::active()->ordered()->get();
        $currentPlan = auth()->user()->currentPlan();
        $subscription = auth()->user()->subscription;

        return view('subscription.plans', compact('plans', 'currentPlan', 'subscription'));
    }

    /**
     * Show checkout page for a plan.
     */
    public function checkout(Plan $plan)
    {
        $user = auth()->user();

        // Check if user already has an active subscription
        if ($user->hasActiveSubscription() && $user->subscription->plan_id === $plan->id) {
            return redirect()->route('subscription.manage')
                ->with('info', 'You are already subscribed to this plan.');
        }

        // Free plan - activate immediately
        if ($plan->isFree()) {
            return $this->activateFreePlan($user, $plan);
        }

        // Create Razorpay order
        try {
            $orderData = $this->razorpayService->createOrder($plan, $user);
            
            return view('subscription.checkout', compact('plan', 'orderData'));
        } catch (\Exception $e) {
            Log::error('Checkout failed: ' . $e->getMessage());
            
            return redirect()->route('subscription.plans')
                ->with('error', 'Unable to process checkout. Please try again later.');
        }
    }

    /**
     * Process payment after Razorpay callback.
     */
    public function processPayment(Request $request)
    {
        $validated = $request->validate([
            'razorpay_payment_id' => 'required|string',
            'razorpay_order_id' => 'required|string',
            'razorpay_signature' => 'required|string',
            'plan_id' => 'required|exists:plans,id',
        ]);

        $user = auth()->user();
        $plan = Plan::findOrFail($validated['plan_id']);

        try {
            DB::beginTransaction();

            $subscription = $this->razorpayService->processSuccessfulPayment(
                $user,
                $plan,
                $validated['razorpay_payment_id'],
                $validated['razorpay_order_id'],
                $validated['razorpay_signature']
            );

            DB::commit();

            return redirect()->route('subscription.success')
                ->with('success', 'Payment successful! Your subscription is now active.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Payment processing failed: ' . $e->getMessage());

            $this->razorpayService->handleFailedPayment(
                $user,
                $plan,
                $validated['razorpay_order_id'],
                $e->getMessage()
            );

            return redirect()->route('subscription.cancel')
                ->with('error', 'Payment verification failed. Please try again.');
        }
    }

    /**
     * Show payment success page.
     */
    public function success()
    {
        $subscription = auth()->user()->subscription;
        
        return view('subscription.success', compact('subscription'));
    }

    /**
     * Show payment cancelled page.
     */
    public function cancel()
    {
        return view('subscription.cancel');
    }

    /**
     * Show subscription management page.
     */
    public function manage()
    {
        $user = auth()->user();
        $subscription = $user->subscription;
        $currentPlan = $subscription->planModel;
        $availablePlans = Plan::active()->ordered()->get();
        $limits = $subscription->getPlanLimits();
        $invoices = $subscription->invoices()->latest()->take(5)->get();

        return view('subscription.manage', compact(
            'subscription',
            'currentPlan',
            'availablePlans',
            'limits',
            'invoices'
        ));
    }

    /**
     * Change subscription plan (upgrade/downgrade).
     */
    public function changePlan(Request $request, Plan $plan)
    {
        $user = auth()->user();
        $subscription = $user->subscription;

        if (!$subscription || !$subscription->isValid()) {
            return redirect()->route('subscription.plans')
                ->with('error', 'You need an active subscription to change plans.');
        }

        $currentPlan = $subscription->planModel;

        // Same plan
        if ($currentPlan->id === $plan->id) {
            return redirect()->route('subscription.manage')
                ->with('info', 'You are already on this plan.');
        }

        // Free plan to paid plan
        if ($currentPlan->isFree() && !$plan->isFree()) {
            return redirect()->route('subscription.checkout', $plan);
        }

        // Paid plan to free plan (downgrade)
        if (!$currentPlan->isFree() && $plan->isFree()) {
            return $this->downgradeToFree($subscription, $plan);
        }

        // Paid to paid (upgrade/downgrade)
        return $this->changePaidPlan($subscription, $plan);
    }

    /**
     * Cancel subscription.
     */
    public function cancelSubscription(Request $request)
    {
        $user = auth()->user();
        $subscription = $user->subscription;

        if (!$subscription || !$subscription->isValid()) {
            return redirect()->route('subscription.manage')
                ->with('error', 'No active subscription to cancel.');
        }

        $immediately = $request->boolean('immediately', false);
        $subscription->cancel($immediately);

        $message = $immediately 
            ? 'Your subscription has been cancelled immediately.'
            : 'Your subscription will be cancelled at the end of the current billing period.';

        return redirect()->route('subscription.manage')
            ->with('success', $message);
    }

    /**
     * Resume cancelled subscription.
     */
    public function resumeSubscription()
    {
        $user = auth()->user();
        $subscription = $user->subscription;

        if (!$subscription || !$subscription->isCancelled()) {
            return redirect()->route('subscription.manage')
                ->with('error', 'No cancelled subscription to resume.');
        }

        $subscription->resume();

        return redirect()->route('subscription.manage')
            ->with('success', 'Your subscription has been resumed successfully.');
    }

    /**
     * Activate free plan.
     */
    protected function activateFreePlan($user, Plan $plan)
    {
        try {
            DB::beginTransaction();

            $subscription = Subscription::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'tenant_id' => $user->tenant_id,
                ],
                [
                    'plan_id' => $plan->id,
                    'plan' => $plan->slug,
                    'status' => Subscription::STATUS_ACTIVE,
                    'current_period_start' => now(),
                    'current_period_end' => now()->addYear(), // Free plans don't expire
                ]
            );

            DB::commit();

            return redirect()->route('subscription.manage')
                ->with('success', 'You are now on the ' . $plan->name . ' plan!');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Free plan activation failed: ' . $e->getMessage());

            return redirect()->route('subscription.plans')
                ->with('error', 'Unable to activate plan. Please try again.');
        }
    }

    /**
     * Downgrade to free plan.
     */
    protected function downgradeToFree(Subscription $subscription, Plan $plan)
    {
        try {
            DB::beginTransaction();

            // Cancel current subscription at period end
            $subscription->cancel(immediately: false);

            // Create new free subscription to start after current period
            // In production, you might want to schedule this

            DB::commit();

            return redirect()->route('subscription.manage')
                ->with('success', 'Your plan will be downgraded to ' . $plan->name . ' at the end of the current billing period.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Downgrade failed: ' . $e->getMessage());

            return redirect()->route('subscription.manage')
                ->with('error', 'Unable to downgrade plan. Please try again.');
        }
    }

    /**
     * Change between paid plans.
     */
    protected function changePaidPlan(Subscription $subscription, Plan $newPlan)
    {
        $currentPlan = $subscription->planModel;
        $isUpgrade = $newPlan->price > $currentPlan->price;

        try {
            DB::beginTransaction();

            // For upgrades, charge immediately
            // For downgrades, apply at period end
            if ($isUpgrade) {
                // Calculate prorated amount
                $daysRemaining = $subscription->daysRemaining();
                $daysInPeriod = 30; // Assuming monthly
                $proratedAmount = ($newPlan->price - $currentPlan->price) * ($daysRemaining / $daysInPeriod);

                // In production, create Razorpay order for prorated amount
                // For now, just update the plan
                $subscription->update([
                    'plan_id' => $newPlan->id,
                    'plan' => $newPlan->slug,
                ]);

                DB::commit();

                return redirect()->route('subscription.manage')
                    ->with('success', 'Your plan has been upgraded to ' . $newPlan->name . '!');
            } else {
                // Downgrade at period end
                $subscription->update([
                    'metadata' => array_merge($subscription->metadata ?? [], [
                        'scheduled_plan_change' => [
                            'plan_id' => $newPlan->id,
                            'plan_slug' => $newPlan->slug,
                            'effective_date' => $subscription->current_period_end,
                        ]
                    ])
                ]);

                DB::commit();

                return redirect()->route('subscription.manage')
                    ->with('success', 'Your plan will be changed to ' . $newPlan->name . ' at the end of the current billing period.');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Plan change failed: ' . $e->getMessage());

            return redirect()->route('subscription.manage')
                ->with('error', 'Unable to change plan. Please try again.');
        }
    }
}