<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\RazorpaySetting;
use Illuminate\Http\Request;

class RazorpaySettingController extends Controller
{
    /**
     * Display the Razorpay settings form.
     */
    public function index()
    {
        $settings = RazorpaySetting::getOrCreate();
        
        return view('superadmin.razorpay.settings', compact('settings'));
    }

    /**
     * Update the Razorpay settings.
     */
    public function update(Request $request)
    {
        $validated = $request->validate([
            'key_id' => 'required|string',
            'key_secret' => 'required|string',
            'webhook_secret' => 'nullable|string',
            'test_mode' => 'boolean',
            'currency' => 'required|string|max:3',
            'is_active' => 'boolean',
        ]);

        $settings = RazorpaySetting::getOrCreate();
        
        // Only update if values are provided (not masked)
        if ($request->filled('key_id') && $request->key_id !== '••••••••••••••••') {
            $settings->key_id = $validated['key_id'];
        }
        
        if ($request->filled('key_secret') && $request->key_secret !== '••••••••••••••••') {
            $settings->key_secret = $validated['key_secret'];
        }
        
        $settings->webhook_secret = $validated['webhook_secret'] ?? $settings->webhook_secret;
        $settings->test_mode = $request->boolean('test_mode');
        $settings->currency = $validated['currency'];
        $settings->is_active = $request->boolean('is_active');
        
        $settings->save();

        return redirect()->route('superadmin.razorpay.settings')
            ->with('success', 'Razorpay settings updated successfully.');
    }

    /**
     * Test the Razorpay connection.
     */
    public function test()
    {
        $settings = RazorpaySetting::getInstance();
        
        if (!$settings || !$settings->isConfigured()) {
            return response()->json([
                'success' => false,
                'message' => 'Razorpay is not configured.'
            ], 400);
        }

        try {
            $credentials = $settings->getCredentials();
            $api = new \Razorpay\Api\Api($credentials['key_id'], $credentials['key_secret']);
            
            // Try to fetch a dummy payment to test credentials
            $api->payment->all(['count' => 1]);
            
            return response()->json([
                'success' => true,
                'message' => 'Razorpay connection successful!'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Razorpay connection failed: ' . $e->getMessage()
            ], 400);
        }
    }
}