<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\RegistrationRequest;
use App\Models\Invite;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class RegistrationRequestController extends Controller
{
    /**
     * Display a listing of registration requests.
     */
    public function index()
    {
        $requests = RegistrationRequest::with(['tenant', 'processedBy'])->orderBy('created_at', 'desc')->get();
        return view('superadmin.requests.index', compact('requests'));
    }

    /**
     * Approve a registration request.
     */
    public function approve(Request $request, RegistrationRequest $registrationRequest)
    {
        // Check if request is already processed
        if ($registrationRequest->status !== 'pending') {
            return redirect()->back()->with('error', 'Request already processed.');
        }
        
        // Create tenant if needed (for brand_admin)
        $tenant = null;
        if ($registrationRequest->requested_role === 'brand_admin') {
            $meta = $registrationRequest->meta;
            $tenant = Tenant::create([
                'name' => $meta['company_name'] ?? $registrationRequest->name . "'s Company",
                'domain' => Str::slug($meta['company_name'] ?? $registrationRequest->name) . '-' . Str::random(6),
            ]);
        }
        
        // Create invite
        $invite = Invite::create([
            'tenant_id' => $tenant ? $tenant->id : null,
            'email' => $registrationRequest->email,
            'token' => Str::random(32),
            'role' => $registrationRequest->requested_role,
            'expires_at' => now()->addDays(7), // Expires in 7 days
        ]);
        
        // Update registration request
        $registrationRequest->update([
            'status' => 'approved',
            'tenant_id' => $tenant ? $tenant->id : null,
            'processed_by' => auth()->id(),
        ]);
        
        // Log the approval
        \App\Models\AuditLog::create([
            'actor_id' => auth()->id(),
            'action' => 'registration_request_approved',
            'target_type' => 'registration_request',
            'target_id' => $registrationRequest->id,
            'meta' => [
                'invite_token' => $invite->token,
                'tenant_id' => $tenant ? $tenant->id : null,
            ]
        ]);
        
        return redirect()->back()->with('status', 'Request approved. Invite has been created and can be sent to the user.');
    }

    /**
     * Reject a registration request.
     */
    public function reject(Request $request, RegistrationRequest $registrationRequest)
    {
        // Check if request is already processed
        if ($registrationRequest->status !== 'pending') {
            return redirect()->back()->with('error', 'Request already processed.');
        }
        
        // Update registration request
        $registrationRequest->update([
            'status' => 'rejected',
            'processed_by' => auth()->id(),
        ]);
        
        // Log the rejection
        \App\Models\AuditLog::create([
            'actor_id' => auth()->id(),
            'action' => 'registration_request_rejected',
            'target_type' => 'registration_request',
            'target_id' => $registrationRequest->id,
            'meta' => []
        ]);
        
        return redirect()->back()->with('status', 'Request rejected.');
    }
}