<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\SubscriptionTransaction;
use App\Models\User;
use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SubscriptionManagementController extends Controller
{
    /**
     * Display a listing of all subscriptions.
     */
    public function index(Request $request)
    {
        $query = Subscription::with(['user', 'planModel', 'tenant']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by plan
        if ($request->filled('plan_id')) {
            $query->where('plan_id', $request->plan_id);
        }

        // Search by user
        if ($request->filled('search')) {
            $query->whereHas('user', function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%');
            });
        }

        $subscriptions = $query->latest()->paginate(20);
        $plans = Plan::active()->ordered()->get();

        // Statistics
        $stats = [
            'total' => Subscription::count(),
            'active' => Subscription::active()->count(),
            'trialing' => Subscription::where('status', Subscription::STATUS_TRIALING)->count(),
            'cancelled' => Subscription::where('status', Subscription::STATUS_CANCELLED)->count(),
            'mrr' => $this->calculateMRR(),
        ];

        return view('superadmin.subscriptions.index', compact('subscriptions', 'plans', 'stats'));
    }

    /**
     * Display the specified subscription.
     */
    public function show(Subscription $subscription)
    {
        $subscription->load(['user', 'planModel', 'tenant', 'transactions', 'invoices']);
        
        return view('superadmin.subscriptions.show', compact('subscription'));
    }

    /**
     * Show form to manually create a subscription.
     */
    public function create()
    {
        $users = User::whereDoesntHave('subscription', function ($query) {
            $query->valid();
        })->get();
        
        $plans = Plan::active()->ordered()->get();

        return view('superadmin.subscriptions.create', compact('users', 'plans'));
    }

    /**
     * Store a manually created subscription.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'plan_id' => 'required|exists:plans,id',
            'status' => 'required|in:active,trialing',
            'trial_days' => 'nullable|integer|min:0',
        ]);

        $user = User::findOrFail($validated['user_id']);
        $plan = Plan::findOrFail($validated['plan_id']);

        $trialDays = $validated['trial_days'] ?? $plan->trial_days;

        $subscription = Subscription::create([
            'user_id' => $user->id,
            'tenant_id' => $user->tenant_id,
            'plan_id' => $plan->id,
            'plan' => $plan->slug,
            'status' => $validated['status'],
            'trial_ends_at' => $trialDays > 0 ? now()->addDays($trialDays) : null,
            'current_period_start' => now(),
            'current_period_end' => now()->addMonth(),
        ]);

        return redirect()->route('superadmin.subscriptions.show', $subscription)
            ->with('success', 'Subscription created successfully.');
    }

    /**
     * Force upgrade/downgrade a subscription.
     */
    public function changePlan(Request $request, Subscription $subscription)
    {
        $validated = $request->validate([
            'plan_id' => 'required|exists:plans,id',
        ]);

        $newPlan = Plan::findOrFail($validated['plan_id']);
        $oldPlan = $subscription->planModel;

        DB::transaction(function () use ($subscription, $newPlan, $oldPlan) {
            // Create transaction record
            SubscriptionTransaction::create([
                'subscription_id' => $subscription->id,
                'user_id' => $subscription->user_id,
                'type' => $newPlan->price > $oldPlan->price 
                    ? SubscriptionTransaction::TYPE_UPGRADE 
                    : SubscriptionTransaction::TYPE_DOWNGRADE,
                'amount' => $newPlan->price,
                'currency' => $newPlan->currency,
                'status' => SubscriptionTransaction::STATUS_SUCCESS,
                'description' => "Plan changed from {$oldPlan->name} to {$newPlan->name} by Super Admin",
            ]);

            // Update subscription
            $subscription->update([
                'plan_id' => $newPlan->id,
                'plan' => $newPlan->slug,
            ]);
        });

        return redirect()->route('superadmin.subscriptions.show', $subscription)
            ->with('success', 'Subscription plan changed successfully.');
    }

    /**
     * Cancel a subscription.
     */
    public function cancel(Subscription $subscription)
    {
        $subscription->cancel(immediately: false);

        return redirect()->route('superadmin.subscriptions.show', $subscription)
            ->with('success', 'Subscription cancelled successfully. It will remain active until the end of the current period.');
    }

    /**
     * Cancel a subscription immediately.
     */
    public function cancelImmediately(Subscription $subscription)
    {
        $subscription->cancel(immediately: true);

        return redirect()->route('superadmin.subscriptions.show', $subscription)
            ->with('success', 'Subscription cancelled immediately.');
    }

    /**
     * Resume a cancelled subscription.
     */
    public function resume(Subscription $subscription)
    {
        if (!$subscription->isCancelled()) {
            return redirect()->back()
                ->with('error', 'Only cancelled subscriptions can be resumed.');
        }

        $subscription->resume();

        return redirect()->route('superadmin.subscriptions.show', $subscription)
            ->with('success', 'Subscription resumed successfully.');
    }

    /**
     * View all transactions.
     */
    public function transactions(Request $request)
    {
        $query = SubscriptionTransaction::with(['subscription.user', 'subscription.planModel']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Date range filter
        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->from_date);
        }

        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->to_date);
        }

        $transactions = $query->latest()->paginate(20);

        // Statistics
        $stats = [
            'total_revenue' => SubscriptionTransaction::successful()
                ->where('type', SubscriptionTransaction::TYPE_PAYMENT)
                ->sum('amount'),
            'total_transactions' => SubscriptionTransaction::count(),
            'successful' => SubscriptionTransaction::successful()->count(),
            'failed' => SubscriptionTransaction::failed()->count(),
        ];

        return view('superadmin.subscriptions.transactions', compact('transactions', 'stats'));
    }

    /**
     * Analytics dashboard.
     */
    public function analytics()
    {
        $stats = [
            'total_subscriptions' => Subscription::count(),
            'active_subscriptions' => Subscription::active()->count(),
            'trialing_subscriptions' => Subscription::where('status', Subscription::STATUS_TRIALING)->count(),
            'cancelled_subscriptions' => Subscription::where('status', Subscription::STATUS_CANCELLED)->count(),
            'mrr' => $this->calculateMRR(),
            'arr' => $this->calculateMRR() * 12,
            'churn_rate' => $this->calculateChurnRate(),
        ];

        // Revenue by plan
        $revenueByPlan = DB::table('subscriptions')
            ->join('plans', 'subscriptions.plan_id', '=', 'plans.id')
            ->select('plans.name', DB::raw('COUNT(*) as count'), DB::raw('SUM(plans.price) as revenue'))
            ->where('subscriptions.status', Subscription::STATUS_ACTIVE)
            ->groupBy('plans.id', 'plans.name')
            ->get();

        // Monthly revenue trend (last 12 months)
        $monthlyRevenue = SubscriptionTransaction::successful()
            ->where('type', SubscriptionTransaction::TYPE_PAYMENT)
            ->where('created_at', '>=', now()->subMonths(12))
            ->selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, SUM(amount) as revenue')
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        return view('superadmin.subscriptions.analytics', compact('stats', 'revenueByPlan', 'monthlyRevenue'));
    }

    /**
     * Calculate Monthly Recurring Revenue (MRR).
     */
    protected function calculateMRR(): float
    {
        return Subscription::active()
            ->join('plans', 'subscriptions.plan_id', '=', 'plans.id')
            ->sum('plans.price');
    }

    /**
     * Calculate churn rate.
     */
    protected function calculateChurnRate(): float
    {
        $startOfMonth = now()->startOfMonth();
        $activeAtStart = Subscription::where('created_at', '<', $startOfMonth)
            ->where(function ($q) use ($startOfMonth) {
                $q->whereNull('cancelled_at')
                  ->orWhere('cancelled_at', '>=', $startOfMonth);
            })
            ->count();

        if ($activeAtStart === 0) {
            return 0;
        }

        $cancelledThisMonth = Subscription::whereBetween('cancelled_at', [
            $startOfMonth,
            now()->endOfMonth()
        ])->count();

        return ($cancelledThisMonth / $activeAtStart) * 100;
    }
}