<?php

namespace App\Jobs;

use App\Models\ReelRequest;
use App\Notifications\ReelCompletedNotification;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class HandleReelCallback implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $payload;
    protected $signature;

    /**
     * Create a new job instance.
     */
    public function __construct(array $payload, string $signature)
    {
        $this->payload = $payload;
        $this->signature = $signature;
    }

    /**
     * Execute the job.
     */
    public function handle()
    {
        try {
            // Verify HMAC signature
            if (!$this->verifySignature()) {
                Log::warning('Invalid signature for reel callback', [
                    'payload' => $this->payload,
                    'signature' => $this->signature
                ]);
                return;
            }

            // Validate payload
            $jobId = $this->payload['job_id'] ?? null;
            $status = $this->payload['status'] ?? null;
            $resultUrl = $this->payload['result_url'] ?? null;
            $duration = $this->payload['duration'] ?? null;
            $error = $this->payload['error'] ?? null;

            if (!$jobId || !$status) {
                Log::warning('Invalid payload for reel callback', ['payload' => $this->payload]);
                return;
            }

            // Find the reel request by job ID
            $reelRequest = ReelRequest::where('job_id', $jobId)->first();
            
            if (!$reelRequest) {
                Log::warning('Reel request not found for job ID', ['job_id' => $jobId]);
                return;
            }

            // Handle success
            if ($status === 'completed' && $resultUrl) {
                // Download the result file
                $resultPath = $this->downloadResult($resultUrl, $reelRequest->id);
                
                if ($resultPath) {
                    // Update the reel request
                    $reelRequest->update([
                        'status' => 'ready',
                        'result_path' => $resultPath,
                        'duration' => $duration
                    ]);

                    // Notify the user
                    $this->notifyUser($reelRequest, 'completed');
                } else {
                    // Mark as failed if download failed
                    $reelRequest->markFailed('Failed to download result file');
                    $this->notifyUser($reelRequest, 'failed', 'Failed to download result file');
                }
            }
            // Handle failure
            elseif ($status === 'failed') {
                $errorMessage = $error ?? 'Processing failed';
                $reelRequest->markFailed($errorMessage);
                $this->notifyUser($reelRequest, 'failed', $errorMessage);
            }
        } catch (\Exception $e) {
            Log::error('Failed to handle reel callback: ' . $e->getMessage(), [
                'payload' => $this->payload,
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Verify the HMAC signature.
     */
    protected function verifySignature(): bool
    {
        $expectedSignature = hash_hmac('sha256', json_encode($this->payload), config('reels.processor_secret'));
        return hash_equals($expectedSignature, $this->signature);
    }

    /**
     * Download the result file from the processor.
     */
    protected function downloadResult(string $url, int $reelId): ?string
    {
        try {
            $response = Http::get($url);
            
            if ($response->successful()) {
                $filename = 'reels/result_' . $reelId . '_' . time() . '.mp4';
                $disk = config('reels.storage_disk', 'public');
                
                Storage::disk($disk)->put($filename, $response->body());
                
                return $filename;
            }
        } catch (\Exception $e) {
            Log::error('Failed to download reel result: ' . $e->getMessage(), [
                'url' => $url,
                'reel_id' => $reelId
            ]);
        }
        
        return null;
    }

    /**
     * Notify the user about the reel status.
     */
    protected function notifyUser(ReelRequest $reelRequest, string $status, string $message = null)
    {
        try {
            $user = $reelRequest->user;
            
            if ($user) {
                Notification::make()
                    ->name('Reel ' . ucfirst($status))
                    ->message($message ?? 'Your reel has been ' . $status)
                    ->user($user)
                    ->send();
            }
            
            // Also notify brand admin if applicable
            if ($reelRequest->brand) {
                foreach ($reelRequest->brand->users()->whereHas('roles', function ($query) {
                    $query->where('name', 'brand_admin');
                })->get() as $admin) {
                    if ($admin->id !== $user->id) {
                        Notification::make()
                            ->name('Reel ' . ucfirst($status))
                            ->message(($message ?? 'A reel has been ' . $status) . ' for your brand')
                            ->user($admin)
                            ->send();
                    }
                }
            }
        } catch (\Exception $e) {
            Log::error('Failed to send reel notification: ' . $e->getMessage());
        }
    }
}