<?php

namespace App\Jobs;

use App\Models\ScheduledReel;
use App\Models\ReelPostAttempt;
use App\Models\SocialAccount;
use App\Services\Providers\ProviderFactory;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessScheduledReel implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $scheduledReel;

    /**
     * Create a new job instance.
     */
    public function __construct(ScheduledReel $scheduledReel)
    {
        $this->scheduledReel = $scheduledReel;
    }

    /**
     * Execute the job.
     */
    public function handle()
    {
        try {
            // Log start of processing
            Log::info('Starting to process scheduled reel', [
                'scheduled_reel_id' => $this->scheduledReel->id,
                'owner_id' => $this->scheduledReel->owner_id,
                'owner_type' => $this->scheduledReel->owner_type,
                'template' => $this->scheduledReel->template,
                'status' => $this->scheduledReel->status
            ]);
            
            // Mark as posting
            $this->scheduledReel->markPosting();
            
            // Get all social accounts for the owner
            $socialAccounts = $this->getSocialAccounts();
            
            Log::info('Found social accounts for scheduled reel', [
                'scheduled_reel_id' => $this->scheduledReel->id,
                'account_count' => $socialAccounts->count()
            ]);
            
            if ($socialAccounts->isEmpty()) {
                $this->scheduledReel->markFailed('No connected social accounts found');
                Log::warning('No connected social accounts found for scheduled reel', [
                    'scheduled_reel_id' => $this->scheduledReel->id
                ]);
                return;
            }
            
            $successCount = 0;
            $failureCount = 0;
            $results = [];
            
            // Process each social account
            foreach ($socialAccounts as $account) {
                try {
                    Log::info('Processing social account for scheduled reel', [
                        'scheduled_reel_id' => $this->scheduledReel->id,
                        'social_account_id' => $account->id,
                        'provider' => $account->provider
                    ]);
                    
                    // Create post attempt record
                    $attempt = ReelPostAttempt::create([
                        'scheduled_reel_id' => $this->scheduledReel->id,
                        'provider' => $account->provider,
                    ]);
                    
                    // Get provider service
                    $provider = ProviderFactory::make($account->provider);
                    
                    if (!$provider) {
                        Log::warning('Provider not supported', [
                            'scheduled_reel_id' => $this->scheduledReel->id,
                            'provider' => $account->provider
                        ]);
                        
                        $attempt->update([
                            'response_summary' => [
                                'status' => 'failed',
                                'message' => 'Provider not supported'
                            ]
                        ]);
                        $failureCount++;
                        continue;
                    }
                    
                    // Prepare request payload
                    $payload = [
                        'scheduled_reel_id' => $this->scheduledReel->id,
                        'template' => $this->scheduledReel->template,
                        'options' => $this->scheduledReel->options,
                        'source_inputs' => $this->scheduledReel->source_inputs,
                    ];
                    
                    $attempt->update([
                        'request_payload' => $payload
                    ]);
                    
                    // Post the reel
                    $result = $provider->postReel($this->scheduledReel, $account);
                    
                    Log::info('Reel post result', [
                        'scheduled_reel_id' => $this->scheduledReel->id,
                        'social_account_id' => $account->id,
                        'provider' => $account->provider,
                        'result_status' => $result['status'] ?? 'unknown'
                    ]);
                    
                    // Update attempt with response
                    $attempt->update([
                        'response_summary' => $result
                    ]);
                    
                    // Check if successful
                    if (isset($result['status']) && $result['status'] === 'success') {
                        $successCount++;
                        $results[$account->provider] = $result;
                    } else {
                        $failureCount++;
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to post reel to ' . $account->provider . ': ' . $e->getMessage(), [
                        'scheduled_reel_id' => $this->scheduledReel->id,
                        'social_account_id' => $account->id,
                        'trace' => $e->getTraceAsString()
                    ]);
                    
                    // Create attempt record with error
                    ReelPostAttempt::create([
                        'scheduled_reel_id' => $this->scheduledReel->id,
                        'provider' => $account->provider,
                        'response_summary' => [
                            'status' => 'failed',
                            'message' => $e->getMessage()
                        ]
                    ]);
                    
                    $failureCount++;
                }
            }
            
            // Update attempt count
            $this->scheduledReel->increment('attempt_count');
            
            // Update result paths
            $this->scheduledReel->update([
                'result_paths' => $results
            ]);
            
            // Log final results
            Log::info('Finished processing scheduled reel', [
                'scheduled_reel_id' => $this->scheduledReel->id,
                'success_count' => $successCount,
                'failure_count' => $failureCount,
                'attempt_count' => $this->scheduledReel->attempt_count
            ]);
            
            // Mark as posted if at least one success
            if ($successCount > 0) {
                $this->scheduledReel->markPosted();
                Log::info('Scheduled reel marked as posted', [
                    'scheduled_reel_id' => $this->scheduledReel->id
                ]);
            } else {
                $this->scheduledReel->markFailed('Failed to post to any social platform');
                Log::warning('Scheduled reel marked as failed', [
                    'scheduled_reel_id' => $this->scheduledReel->id
                ]);
            }
            
            // Send notifications
            $this->sendNotifications($successCount, $failureCount, $results);
        } catch (\Exception $e) {
            Log::error('Failed to process scheduled reel: ' . $e->getMessage(), [
                'scheduled_reel_id' => $this->scheduledReel->id,
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->scheduledReel->markFailed('Processing failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Get social accounts for the owner.
     */
    protected function getSocialAccounts()
    {
        if ($this->scheduledReel->owner_type === 'App\Models\Brand') {
            // Get all social accounts for the brand's creators
            return SocialAccount::whereHas('user', function ($query) {
                $query->where('brand_id', $this->scheduledReel->owner_id);
            })->where('tenant_id', tenant('id'))->get();
        } elseif ($this->scheduledReel->owner_type === 'App\Models\User') {
            // Get all social accounts for the user
            return SocialAccount::where('user_id', $this->scheduledReel->owner_id)
                ->where('tenant_id', tenant('id'))
                ->get();
        }
        
        return collect();
    }
    
    /**
     * Send notifications about the posting results.
     */
    protected function sendNotifications(int $successCount, int $failureCount, array $results)
    {
        try {
            // Create a notification for the owner
            $owner = $this->scheduledReel->owner;
            
            if ($owner) {
                // Send notification
                $owner->notify(new \App\Notifications\ReelPosted($this->scheduledReel, $successCount, $failureCount));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send notifications for scheduled reel: ' . $e->getMessage(), [
                'scheduled_reel_id' => $this->scheduledReel->id
            ]);
        }
    }
}