<?php

namespace App\Jobs;

use App\Models\SocialAccount;
use App\Models\SocialAggregate;
use App\Models\SocialPost;
use App\Models\SocialSyncError;
use App\Models\User;
use App\Services\SocialProviders\ProviderFactory;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\DB;

class SyncCreatorSocialAccounts implements ShouldQueue
{
    use Queueable;

    /**
     * The creator to sync.
     *
     * @var User
     */
    protected $creator;

    /**
     * The specific account to sync.
     *
     * @var SocialAccount|null
     */
    protected $account;

    /**
     * Create a new job instance.
     *
     * @param User|SocialAccount $creatorOrAccount
     * @return void
     */
    public function __construct($creatorOrAccount = null)
    {
        if ($creatorOrAccount instanceof User) {
            $this->creator = $creatorOrAccount;
        } elseif ($creatorOrAccount instanceof SocialAccount) {
            $this->account = $creatorOrAccount;
        }
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle(): void
    {
        // Determine which accounts to sync
        $accounts = $this->getAccountsToSync();

        // Sync each account
        foreach ($accounts as $account) {
            $this->syncAccount($account);
        }

        // Update aggregates after syncing all accounts
        if ($this->creator) {
            $this->updateAggregates();
        } elseif ($this->account) {
            $this->updateAggregatesForAccount($this->account);
        }
    }

    /**
     * Get the accounts to sync.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    protected function getAccountsToSync()
    {
        if ($this->account) {
            return collect([$this->account]);
        }

        if ($this->creator) {
            return SocialAccount::where('user_id', $this->creator->id)->get();
        }

        return collect();
    }

    /**
     * Sync a single account.
     *
     * @param SocialAccount $account
     * @return void
     */
    protected function syncAccount(SocialAccount $account)
    {
        // For demo accounts, just update the last synced time
        if ($account->is_demo) {
            $account->last_synced_at = now();
            $account->save();
            return;
        }

        try {
            // Get the provider instance
            $provider = ProviderFactory::make($account->provider);

            // Fetch account details
            $details = $provider->fetchAccountDetails($account);

            // Update account with details
            if (!empty($details)) {
                $account->meta = array_merge($account->meta ?? [], $details);
            }

            // Fetch recent posts
            $posts = $provider->fetchRecentPosts($account, 20);

            // Process posts
            $this->processPosts($account, $posts);

            // Update last synced time
            $account->last_synced_at = now();
            $account->save();
        } catch (\Exception $e) {
            // Log the error
            SocialSyncError::create([
                'social_account_id' => $account->id,
                'error_message' => $e->getMessage(),
                'error_trace' => $e->getTraceAsString(),
                'occurred_at' => now(),
            ]);
            
            \Log::error('Failed to sync creator account: ' . $e->getMessage(), [
                'account_id' => $account->id,
                'provider' => $account->provider,
                'user_id' => $account->user_id,
            ]);
        }
    }

    /**
     * Process posts for an account.
     *
     * @param SocialAccount $account
     * @param array $posts
     * @return void
     */
    protected function processPosts(SocialAccount $account, array $posts)
    {
        foreach ($posts as $postData) {
            // Use database transaction for upsert
            DB::transaction(function () use ($account, $postData) {
                // Find existing post or create new one
                $post = SocialPost::updateOrCreate(
                    [
                        'social_account_id' => $account->id,
                        'provider_post_id' => $postData['id'],
                    ],
                    [
                        'title' => $postData['title'] ?? null,
                        'caption' => $postData['caption'] ?? null,
                        'content' => $postData['text'] ?? $postData['description'] ?? null,
                        'media' => [
                            'thumbnail' => $postData['thumbnail'] ?? $postData['media_url'] ?? $postData['image_url'] ?? null,
                            'video' => $postData['video_url'] ?? null,
                        ],
                        'permalink' => $postData['permalink'] ?? $postData['link'] ?? null,
                        'published_at' => $postData['published_at'] ?? $postData['created_at'] ?? $postData['timestamp'] ?? null,
                        'metrics' => [
                            'views' => $postData['views'] ?? 0,
                            'likes' => $postData['likes'] ?? $postData['ups'] ?? $postData['like_count'] ?? 0,
                            'comments' => $postData['comments'] ?? $postData['comment_count'] ?? 0,
                            'shares' => $postData['shares'] ?? $postData['shares'] ?? 0,
                            'saves' => $postData['saves'] ?? $postData['save_count'] ?? 0,
                        ],
                        'raw_json' => $postData,
                        'last_checked_at' => now(),
                    ]
                );
            });
        }
    }

    /**
     * Update aggregates for the creator.
     *
     * @return void
     */
    protected function updateAggregates()
    {
        $userId = $this->creator->id;

        // Calculate total followers
        $totalFollowers = SocialAccount::where('user_id', $userId)
            ->where('is_demo', false)
            ->sum('meta->followers');

        // For demo accounts, add fake data
        $demoFollowers = SocialAccount::where('user_id', $userId)
            ->where('is_demo', true)
            ->sum('meta->followers');

        $totalFollowers += $demoFollowers;

        // Update or create aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'key' => 'total_followers',
            ],
            [
                'value' => [
                    'value' => $totalFollowers,
                    'previous' => $this->getPreviousAggregateValue($userId, 'total_followers'),
                ],
                'computed_at' => now(),
            ]
        );

        // Calculate total posts
        $totalPosts = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->count();

        // Update or create aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'key' => 'total_posts',
            ],
            [
                'value' => [
                    'value' => $totalPosts,
                    'previous' => $this->getPreviousAggregateValue($userId, 'total_posts'),
                ],
                'computed_at' => now(),
            ]
        );

        // Calculate total views
        $totalViews = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->sum('metrics->views');

        // Update or create aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'key' => 'total_views',
            ],
            [
                'value' => [
                    'value' => $totalViews,
                    'previous' => $this->getPreviousAggregateValue($userId, 'total_views'),
                ],
                'computed_at' => now(),
            ]
        );

        // Calculate engagement rate
        $totalLikes = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->sum('metrics->likes');

        $totalComments = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->sum('metrics->comments');
        
        $totalShares = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->sum('metrics->shares');
        
        $totalSaves = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->sum('metrics->saves');

        $engagementRate = 0;
        if ($totalViews > 0) {
            $engagementRate = round((($totalLikes + $totalComments + $totalShares + $totalSaves) / $totalViews) * 100, 2);
        }

        // Update or create aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'key' => 'engagement_rate',
            ],
            [
                'value' => [
                    'value' => $engagementRate,
                    'previous' => $this->getPreviousAggregateValue($userId, 'engagement_rate'),
                ],
                'computed_at' => now(),
            ]
        );

        // Calculate daily views for the last 14 days
        $this->updateDailyViews($userId);
    }

    /**
     * Update aggregates for a specific account.
     *
     * @param SocialAccount $account
     * @return void
     */
    protected function updateAggregatesForAccount(SocialAccount $account)
    {
        // This is a simplified version that just calls the main updateAggregates method
        // In a real implementation, you might want to update only account-specific aggregates
        $this->creator = $account->user;
        $this->updateAggregates();
    }

    /**
     * Get the previous aggregate value.
     *
     * @param int $userId
     * @param string $key
     * @return mixed
     */
    protected function getPreviousAggregateValue($userId, $key)
    {
        $aggregate = SocialAggregate::where('user_id', $userId)
            ->where('key', $key)
            ->first();

        return $aggregate ? $aggregate->getNumericValue() : null;
    }

    /**
     * Update daily views for the last 14 days.
     *
     * @param int $userId
     * @return void
     */
    protected function updateDailyViews($userId)
    {
        // Get posts from the last 14 days
        $posts = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->where('published_at', '>=', now()->subDays(14))
            ->get();

        // Group by day and sum views
        $dailyViews = [];
        foreach ($posts as $post) {
            $day = $post->published_at->format('Y-m-d');
            if (!isset($dailyViews[$day])) {
                $dailyViews[$day] = 0;
            }
            $dailyViews[$day] += $post->getViewsCount();
        }

        // Update or create aggregates for each day
        foreach ($dailyViews as $day => $views) {
            SocialAggregate::updateOrCreate(
                [
                    'user_id' => $userId,
                    'key' => 'daily_views',
                    'computed_at' => $day,
                ],
                [
                    'value' => $views,
                    'computed_at' => $day,
                ]
            );
        }
    }
}