<?php

namespace App\Jobs;

use App\Models\SocialAccount;
use App\Models\SocialAggregate;
use App\Models\SocialPost;
use App\Models\User;
use App\Services\SocialProviders\ProviderFactory;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\DB;

class SyncUserSocialAccounts implements ShouldQueue
{
    use Queueable;

    /**
     * The user ID to sync.
     *
     * @var int|null
     */
    protected $userId;

    /**
     * The specific account ID to sync.
     *
     * @var int|null
     */
    protected $accountId;

    /**
     * Create a new job instance.
     *
     * @param int|SocialAccount $userOrAccountId
     * @return void
     */
    public function __construct($userOrAccountId = null)
    {
        if ($userOrAccountId instanceof User) {
            $this->userId = $userOrAccountId->id;
        } elseif ($userOrAccountId instanceof SocialAccount) {
            $this->accountId = $userOrAccountId->id;
        } elseif (is_int($userOrAccountId)) {
            // Assume it's a user ID
            $this->userId = $userOrAccountId;
        }
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle(): void
    {
        // Determine which accounts to sync
        $accounts = $this->getAccountsToSync();

        // Sync each account
        foreach ($accounts as $account) {
            $this->syncAccount($account);
        }

        // Update aggregates after syncing all accounts
        if ($this->userId) {
            $this->updateAggregates();
        } elseif ($this->accountId) {
            $account = SocialAccount::find($this->accountId);
            if ($account) {
                $this->updateAggregatesForAccount($account);
            }
        }
    }

    /**
     * Get the accounts to sync.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    protected function getAccountsToSync()
    {
        if ($this->accountId) {
            $account = SocialAccount::find($this->accountId);
            return $account ? collect([$account]) : collect();
        }
        
        if ($this->userId) {
            return SocialAccount::where('user_id', $this->userId)->get();
        }
        
        return collect();
    }

    /**
     * Sync a single account.
     *
     * @param SocialAccount $account
     * @return void
     */
    protected function syncAccount(SocialAccount $account)
    {
        // For demo accounts, just update the last synced time
        if ($account->is_demo) {
            $account->last_synced_at = now();
            $account->save();
            return;
        }

        // Guard clause for missing token
        if (!$account->getDecryptedToken() && !$account->isDemo()) {
            // Log error for missing token
            \Log::warning('Missing access token for account: ' . $account->id);
            
            // Record the error in the database
            $error = new \App\Models\SocialSyncError();
            $error->social_account_id = $account->id;
            $error->error_text = 'Missing access token';
            $error->response_code = null;
            $error->meta = [
                'provider' => $account->provider,
            ];
            $error->save();
            
            return;
        }
        
        try {
            // Get the provider instance
            $provider = ProviderFactory::make($account->provider);
            
            // Fetch account details
            $details = $provider->fetchAccountDetails($account);
            
            // Update account with details
            if (!empty($details)) {
                $account->meta = array_merge($account->meta ?? [], $details);
            }
            
            // Fetch recent posts
            $posts = $provider->fetchRecentPosts($account, 20);
            
            // Process posts
            $this->processPosts($account, $posts);
            
            // Update last synced time
            $account->last_synced_at = now();
            $account->save();
        } catch (\Exception $e) {
            // Log the error but continue with other accounts
            \Log::error('Failed to sync account: ' . $e->getMessage(), [
                'account_id' => $account->id,
                'provider' => $account->provider,
            ]);
            
            // Record the error in the database
            $error = new \App\Models\SocialSyncError();
            $error->social_account_id = $account->id;
            $error->error_text = 'Failed to sync account: ' . $e->getMessage();
            $error->response_code = method_exists($e, 'getResponse') ? $e->getResponse()->getStatusCode() : null;
            $error->meta = [
                'exception' => get_class($e),
                'provider' => $account->provider,
            ];
            $error->save();
        }
    }

    /**
     * Process posts for an account.
     *
     * @param SocialAccount $account
     * @param array $posts
     * @return void
     */
    protected function processPosts(SocialAccount $account, array $posts)
    {
        foreach ($posts as $postData) {
            // Guard clause for missing post ID
            if (!isset($postData['id'])) {
                \Log::warning('Missing post ID in data for account: ' . $account->id);
                continue;
            }
            
            try {
                // Use database transaction for upsert
                DB::transaction(function () use ($account, $postData) {
                    // Find existing post or create new one
                    $post = SocialPost::updateOrCreate(
                        [
                            'social_account_id' => $account->id,
                            'provider_post_id' => $postData['id'],
                        ],
                        [
                            'title' => $postData['title'] ?? null,
                            'caption' => $postData['caption'] ?? null,
                            'content' => $postData['text'] ?? $postData['description'] ?? null,
                            'media' => [
                                'thumbnail' => $postData['thumbnail'] ?? $postData['media_url'] ?? $postData['image_url'] ?? null,
                                'video' => $postData['video_url'] ?? null,
                            ],
                            'permalink' => $postData['permalink'] ?? $postData['link'] ?? null,
                            'published_at' => $postData['published_at'] ?? $postData['created_at'] ?? $postData['timestamp'] ?? null,
                            'metrics' => [
                                'views' => $postData['views'] ?? 0,
                                'likes' => $postData['likes'] ?? $postData['ups'] ?? $postData['like_count'] ?? 0,
                                'comments' => $postData['comments'] ?? $postData['comment_count'] ?? 0,
                                'shares' => $postData['shares'] ?? $postData['shares'] ?? 0,
                            ],
                            'raw_json' => $postData,
                            'last_checked_at' => now(),
                        ]
                    );
                });
            } catch (\Exception $e) {
                // Log the error but continue with other posts
                \Log::error('Failed to process post for account ' . $account->id . ': ' . $e->getMessage(), [
                    'post_id' => $postData['id'] ?? null,
                    'provider' => $account->provider,
                ]);
            }
        }
    }

    /**
     * Update aggregates for the user.
     *
     * @return void
     */
    protected function updateAggregates()
    {
        $userId = $this->userId;

        // Calculate total followers
        $totalFollowers = SocialAccount::where('user_id', $userId)
            ->where('is_demo', false)
            ->sum('meta->followers');

        // For demo accounts, add fake data
        $demoFollowers = SocialAccount::where('user_id', $userId)
            ->where('is_demo', true)
            ->sum('meta->followers');

        $totalFollowers += $demoFollowers;

        // Update or create aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'key' => 'total_followers',
            ],
            [
                'value' => [
                    'value' => $totalFollowers,
                    'previous' => $this->getPreviousAggregateValue($userId, 'total_followers'),
                ],
                'computed_at' => now(),
            ]
        );

        // Calculate total posts
        $totalPosts = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->count();

        // Update or create aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'key' => 'total_posts',
            ],
            [
                'value' => [
                    'value' => $totalPosts,
                    'previous' => $this->getPreviousAggregateValue($userId, 'total_posts'),
                ],
                'computed_at' => now(),
            ]
        );

        // Calculate total views
        $totalViews = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->sum('metrics->views');

        // Update or create aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'key' => 'total_views',
            ],
            [
                'value' => [
                    'value' => $totalViews,
                    'previous' => $this->getPreviousAggregateValue($userId, 'total_views'),
                ],
                'computed_at' => now(),
            ]
        );

        // Calculate engagement rate
        $totalLikes = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->sum('metrics->likes');

        $totalComments = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->sum('metrics->comments');

        $engagementRate = 0;
        if ($totalViews > 0) {
            $engagementRate = round((($totalLikes + $totalComments) / $totalViews) * 100, 2);
        }

        // Update or create aggregate
        SocialAggregate::updateOrCreate(
            [
                'user_id' => $userId,
                'key' => 'engagement_rate',
            ],
            [
                'value' => [
                    'value' => $engagementRate,
                    'previous' => $this->getPreviousAggregateValue($userId, 'engagement_rate'),
                ],
                'computed_at' => now(),
            ]
        );

        // Calculate daily views for the last 14 days
        $this->updateDailyViews($userId);
    }

    /**
     * Update aggregates for a specific account.
     *
     * @param SocialAccount $account
     * @return void
     */
    protected function updateAggregatesForAccount(SocialAccount $account)
    {
        // This is a simplified version that just calls the main updateAggregates method
        // In a real implementation, you might want to update only account-specific aggregates
        $this->userId = $account->user_id;
        $this->updateAggregates();
    }

    /**
     * Get the previous aggregate value.
     *
     * @param int $userId
     * @param string $key
     * @return mixed
     */
    protected function getPreviousAggregateValue($userId, $key)
    {
        $aggregate = SocialAggregate::where('user_id', $userId)
            ->where('key', $key)
            ->first();

        return $aggregate ? $aggregate->getNumericValue() : null;
    }

    /**
     * Update daily views for the last 14 days.
     *
     * @param int $userId
     * @return void
     */
    protected function updateDailyViews($userId)
    {
        try {
            // Get posts from the last 14 days
            $posts = SocialPost::whereHas('socialAccount', function ($query) use ($userId) {
                $query->where('user_id', $userId);
            })->where('published_at', '>=', now()->subDays(14))
                ->get();
            
            // Group by day and sum views
            $dailyViews = [];
            foreach ($posts as $post) {
                // Guard clause for null published_at
                if (!$post->published_at) {
                    continue;
                }
                
                $day = $post->published_at->format('Y-m-d');
                if (!isset($dailyViews[$day])) {
                    $dailyViews[$day] = 0;
                }
                $dailyViews[$day] += $post->getViewsCount();
            }
            
            // Update or create aggregates for each day
            foreach ($dailyViews as $day => $views) {
                SocialAggregate::updateOrCreate(
                    [
                        'user_id' => $userId,
                        'key' => 'daily_views',
                        'computed_at' => $day,
                    ],
                    [
                        'value' => $views,
                        'computed_at' => $day,
                    ]
                );
            }
        } catch (\Exception $e) {
            // Log the error but continue
            \Log::error('Failed to update daily views for user ' . $userId . ': ' . $e->getMessage());
        }
    }
}
