<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BudgetScenario extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'brand_id',
        'financial_report_id',
        'name',
        'scenario_type',
        'description',
        'total_budget',
        'channel_allocation',
        'campaign_allocation',
        'projected_revenue',
        'projected_roi',
        'projected_conversions',
        'projected_cost_per_conversion',
        'ai_recommendations',
        'risk_assessment',
        'optimization_suggestions',
        'roi_improvement',
        'revenue_increase',
        'status',
        'is_applied',
        'applied_at',
    ];

    protected $casts = [
        'total_budget' => 'decimal:2',
        'channel_allocation' => 'array',
        'campaign_allocation' => 'array',
        'projected_revenue' => 'decimal:2',
        'projected_roi' => 'decimal:2',
        'projected_conversions' => 'integer',
        'projected_cost_per_conversion' => 'decimal:2',
        'ai_recommendations' => 'array',
        'optimization_suggestions' => 'array',
        'roi_improvement' => 'decimal:2',
        'revenue_increase' => 'decimal:2',
        'is_applied' => 'boolean',
        'applied_at' => 'datetime',
    ];

    /**
     * Get the brand that owns the scenario
     */
    public function brand(): BelongsTo
    {
        return $this->belongsTo(User::class, 'brand_id');
    }

    /**
     * Get the financial report associated with this scenario
     */
    public function financialReport(): BelongsTo
    {
        return $this->belongsTo(FinancialReport::class);
    }

    /**
     * Scope to filter by scenario type
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('scenario_type', $type);
    }

    /**
     * Scope to filter by brand
     */
    public function scopeForBrand($query, $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Scope to filter by status
     */
    public function scopeWithStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope to get active scenarios
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope to get applied scenarios
     */
    public function scopeApplied($query)
    {
        return $query->where('is_applied', true);
    }

    /**
     * Mark scenario as applied
     */
    public function markAsApplied(): void
    {
        $this->update([
            'is_applied' => true,
            'applied_at' => now(),
            'status' => 'active',
        ]);
    }

    /**
     * Archive the scenario
     */
    public function archive(): void
    {
        $this->update(['status' => 'archived']);
    }

    /**
     * Activate the scenario
     */
    public function activate(): void
    {
        $this->update(['status' => 'active']);
    }

    /**
     * Get projected profit
     */
    public function getProjectedProfitAttribute(): float
    {
        return $this->projected_revenue - $this->total_budget;
    }

    /**
     * Check if scenario is profitable
     */
    public function isProfitable(): bool
    {
        return $this->projected_profit > 0;
    }

    /**
     * Get risk level based on scenario type
     */
    public function getRiskLevel(): string
    {
        return match($this->scenario_type) {
            'conservative' => 'low',
            'balanced' => 'medium',
            'aggressive' => 'high',
            default => 'unknown',
        };
    }

    /**
     * Get scenario performance rating
     */
    public function getPerformanceRating(): string
    {
        if ($this->projected_roi === null) {
            return 'unknown';
        }

        if ($this->projected_roi > 30) {
            return 'excellent';
        } elseif ($this->projected_roi > 20) {
            return 'very_good';
        } elseif ($this->projected_roi > 10) {
            return 'good';
        } elseif ($this->projected_roi > 0) {
            return 'fair';
        }

        return 'poor';
    }

    /**
     * Get channel allocation percentage for a specific channel
     */
    public function getChannelAllocation(string $channel): float
    {
        if (!is_array($this->channel_allocation)) {
            return 0;
        }

        return $this->channel_allocation[$channel] ?? 0;
    }

    /**
     * Get campaign allocation percentage for a specific campaign
     */
    public function getCampaignAllocation(int $campaignId): float
    {
        if (!is_array($this->campaign_allocation)) {
            return 0;
        }

        return $this->campaign_allocation[$campaignId] ?? 0;
    }

    /**
     * Calculate total channel allocation percentage
     */
    public function getTotalChannelAllocationAttribute(): float
    {
        if (!is_array($this->channel_allocation)) {
            return 0;
        }

        return array_sum($this->channel_allocation);
    }

    /**
     * Validate channel allocation (should sum to 100%)
     */
    public function hasValidChannelAllocation(): bool
    {
        $total = $this->total_channel_allocation;
        return $total >= 99 && $total <= 101; // Allow 1% tolerance
    }
}