<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class LocalizedContent extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'brand_id',
        'content_type',
        'content_id',
        'locale_code',
        'source_text',
        'localized_text',
        'cultural_notes',
        'tone',
        'ai_job_id',
        'status',
        'approved_by',
        'approved_at',
        'created_by',
    ];

    protected $casts = [
        'approved_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * Get the brand that owns the localized content.
     */
    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the user who created the localization request.
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who approved the localization.
     */
    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Get the audit logs for this localized content.
     */
    public function auditLogs(): HasMany
    {
        return $this->hasMany(LocalizationAuditLog::class);
    }

    /**
     * Get the content that is being localized (polymorphic).
     */
    public function content(): MorphTo
    {
        return $this->morphTo(__FUNCTION__, 'content_type', 'content_id');
    }

    /**
     * Scope to filter by locale.
     */
    public function scopeForLocale($query, string $locale)
    {
        return $query->where('locale_code', $locale);
    }

    /**
     * Scope to filter by status.
     */
    public function scopeWithStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope to filter by brand.
     */
    public function scopeForBrand($query, int $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Scope to filter by content type.
     */
    public function scopeForContentType($query, string $contentType)
    {
        return $query->where('content_type', $contentType);
    }

    /**
     * Check if the localization is approved.
     */
    public function isApproved(): bool
    {
        return $this->status === 'approved';
    }

    /**
     * Check if the localization is pending.
     */
    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    /**
     * Check if the localization is generated.
     */
    public function isGenerated(): bool
    {
        return $this->status === 'generated';
    }

    /**
     * Approve the localization.
     */
    public function approve(User $user): bool
    {
        $this->status = 'approved';
        $this->approved_by = $user->id;
        $this->approved_at = now();
        
        return $this->save();
    }

    /**
     * Reject the localization.
     */
    public function reject(): bool
    {
        $this->status = 'rejected';
        
        return $this->save();
    }

    /**
     * Mark as edited.
     */
    public function markAsEdited(): bool
    {
        $this->status = 'edited';
        
        return $this->save();
    }

    /**
     * Get locale display name.
     */
    public function getLocaleDisplayNameAttribute(): string
    {
        $localeNames = [
            'en' => 'English',
            'en-US' => 'English (US)',
            'en-GB' => 'English (UK)',
            'es' => 'Spanish',
            'es-ES' => 'Spanish (Spain)',
            'es-MX' => 'Spanish (Mexico)',
            'fr' => 'French',
            'fr-FR' => 'French (France)',
            'de' => 'German',
            'de-DE' => 'German (Germany)',
            'it' => 'Italian',
            'it-IT' => 'Italian (Italy)',
            'pt' => 'Portuguese',
            'pt-BR' => 'Portuguese (Brazil)',
            'pt-PT' => 'Portuguese (Portugal)',
            'ja' => 'Japanese',
            'ja-JP' => 'Japanese (Japan)',
            'zh' => 'Chinese',
            'zh-CN' => 'Chinese (Simplified)',
            'zh-TW' => 'Chinese (Traditional)',
            'ko' => 'Korean',
            'ko-KR' => 'Korean (Korea)',
            'ar' => 'Arabic',
            'ar-SA' => 'Arabic (Saudi Arabia)',
            'hi' => 'Hindi',
            'hi-IN' => 'Hindi (India)',
        ];

        return $localeNames[$this->locale_code] ?? $this->locale_code;
    }
}