<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Message extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'sender_id',
        'recipient_id',
        'subject',
        'message',
        'attachment_path',
        'is_read',
        'read_at',
        'parent_id',
        'thread_id',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_read' => 'boolean',
        'read_at' => 'datetime',
    ];

    /**
     * Get the sender of the message.
     */
    public function sender()
    {
        return $this->belongsTo(User::class, 'sender_id');
    }

    /**
     * Get the recipient of the message.
     */
    public function recipient()
    {
        return $this->belongsTo(User::class, 'recipient_id');
    }

    /**
     * Get the thread that this message belongs to.
     */
    public function thread()
    {
        return $this->belongsTo(MessageThread::class, 'thread_id');
    }

    /**
     * Get the parent message.
     */
    public function parent()
    {
        return $this->belongsTo(Message::class, 'parent_id');
    }

    /**
     * Get the child messages.
     */
    public function children()
    {
        return $this->hasMany(Message::class, 'parent_id');
    }
    
    /**
     * Scope a query to only include messages sent by a specific user.
     */
    public function scopeSentBy($query, $userId)
    {
        return $query->where('sender_id', $userId);
    }
    
    /**
     * Scope a query to only include messages received by a specific user.
     */
    public function scopeReceivedBy($query, $userId)
    {
        return $query->where('recipient_id', $userId);
    }
    
    /**
     * Scope a query to only include unread messages.
     */
    public function scopeUnread($query)
    {
        return $query->where('is_read', false);
    }
    
    /**
     * Get the unread messages count for a user.
     */
    public static function unreadCount($userId)
    {
        return static::receivedBy($userId)->unread()->count();
    }
}