<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class ScheduledReel extends Model
{
    use HasFactory;

    protected $fillable = [
        'owner_id',
        'owner_type',
        'poster_id',
        'template',
        'options',
        'source_inputs',
        'scheduled_at',
        'timezone',
        'status',
        'processor_job_id',
        'result_paths',
        'attempt_count',
        'error_message',
        'created_by',
    ];

    protected $casts = [
        'options' => 'array',
        'source_inputs' => 'array',
        'result_paths' => 'array',
        'scheduled_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the owner (brand or creator) of the scheduled reel.
     */
    public function owner()
    {
        return $this->morphTo();
    }

    /**
     * Get the poster associated with the scheduled reel.
     */
    public function poster()
    {
        return $this->belongsTo(Poster::class);
    }

    /**
     * Get the post attempts for this scheduled reel.
     */
    public function postAttempts()
    {
        return $this->hasMany(ReelPostAttempt::class, 'scheduled_reel_id');
    }

    /**
     * Get the user who created this scheduled reel.
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Check if the scheduled reel is due for posting.
     */
    public function isDue(): bool
    {
        return $this->scheduled_at && $this->scheduled_at->isPast() && $this->status === 'scheduled';
    }

    /**
     * Mark the scheduled reel as queued.
     */
    public function markQueued(): void
    {
        $this->status = 'queued';
        $this->save();
    }

    /**
     * Mark the scheduled reel as posting.
     */
    public function markPosting(): void
    {
        $this->status = 'posting';
        $this->save();
    }

    /**
     * Mark the scheduled reel as posted.
     */
    public function markPosted(): void
    {
        $this->status = 'posted';
        $this->save();
    }

    /**
     * Mark the scheduled reel as failed.
     */
    public function markFailed(string $message): void
    {
        $this->status = 'failed';
        $this->error_message = $message;
        $this->save();
    }

    /**
     * Mark the scheduled reel as cancelled.
     */
    public function markCancelled(): void
    {
        $this->status = 'cancelled';
        $this->save();
    }

    /**
     * Get signed URLs for all source inputs.
     */
    public function getSignedSourceUrls(): array
    {
        $urls = [];
        
        if (!is_array($this->source_inputs)) {
            return $urls;
        }
        
        foreach ($this->source_inputs as $input) {
            if (isset($input['source_url'])) {
                $urls[] = $input['source_url'];
            } elseif (isset($input['source_path']) && Storage::disk('public')->exists($input['source_path'])) {
                $expireMinutes = config('reels.signed_url_expire', 300);
                $urls[] = Storage::disk('public')->temporaryUrl(
                    $input['source_path'],
                    now()->addMinutes($expireMinutes)
                );
            }
        }
        
        return $urls;
    }
}