<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SubscriptionFeatureUsage extends Model
{
    protected $table = 'subscription_feature_usage';

    protected $fillable = [
        'subscription_id',
        'user_id',
        'feature',
        'used',
        'limit',
        'period_start',
        'period_end',
    ];

    protected $casts = [
        'used' => 'integer',
        'limit' => 'integer',
        'period_start' => 'date',
        'period_end' => 'date',
    ];

    /**
     * Get the subscription that owns the feature usage.
     */
    public function subscription(): BelongsTo
    {
        return $this->belongsTo(Subscription::class);
    }

    /**
     * Get the user that owns the feature usage.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Check if limit is reached.
     */
    public function isLimitReached(): bool
    {
        if ($this->limit === -1) {
            return false; // Unlimited
        }

        return $this->used >= $this->limit;
    }

    /**
     * Check if feature is unlimited.
     */
    public function isUnlimited(): bool
    {
        return $this->limit === -1;
    }

    /**
     * Get remaining usage.
     */
    public function getRemainingAttribute(): int
    {
        if ($this->isUnlimited()) {
            return -1;
        }

        return max(0, $this->limit - $this->used);
    }

    /**
     * Get usage percentage.
     */
    public function getUsagePercentageAttribute(): float
    {
        if ($this->isUnlimited() || $this->limit === 0) {
            return 0;
        }

        return min(100, ($this->used / $this->limit) * 100);
    }

    /**
     * Increment usage.
     */
    public function increment(int $amount = 1): bool
    {
        $this->used += $amount;
        return $this->save();
    }

    /**
     * Reset usage.
     */
    public function reset(): bool
    {
        $this->used = 0;
        return $this->save();
    }

    /**
     * Scope to get current period usage.
     */
    public function scopeCurrentPeriod($query)
    {
        $now = now();
        return $query->where('period_start', '<=', $now)
                     ->where('period_end', '>=', $now);
    }

    /**
     * Scope to get usage by feature.
     */
    public function scopeByFeature($query, string $feature)
    {
        return $query->where('feature', $feature);
    }
}