<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class SupportConversation extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'conversation_id',
        'brand_id',
        'customer_id',
        'customer_name',
        'customer_email',
        'customer_phone',
        'channel',
        'status',
        'initial_query',
        'metadata',
        'escalated_to_human',
        'assigned_agent_id',
        'escalated_at',
        'resolved_at',
        'closed_at',
        'total_messages',
        'avg_confidence_score',
        'satisfaction_rating',
        'satisfaction_feedback',
    ];

    protected $casts = [
        'metadata' => 'array',
        'escalated_to_human' => 'boolean',
        'escalated_at' => 'datetime',
        'resolved_at' => 'datetime',
        'closed_at' => 'datetime',
        'avg_confidence_score' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($conversation) {
            if (!$conversation->conversation_id) {
                $conversation->conversation_id = 'conv_' . Str::random(16);
            }
        });
    }

    /**
     * Get the brand that owns the conversation.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the customer (if registered user).
     */
    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    /**
     * Get the assigned agent.
     */
    public function assignedAgent()
    {
        return $this->belongsTo(User::class, 'assigned_agent_id');
    }

    /**
     * Get all messages in the conversation.
     */
    public function messages()
    {
        return $this->hasMany(SupportMessage::class, 'conversation_id');
    }

    /**
     * Scope to get active conversations.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope to get escalated conversations.
     */
    public function scopeEscalated($query)
    {
        return $query->where('escalated_to_human', true);
    }

    /**
     * Scope to get resolved conversations.
     */
    public function scopeResolved($query)
    {
        return $query->where('status', 'resolved');
    }

    /**
     * Escalate conversation to human agent.
     */
    public function escalate(?int $agentId = null, ?string $reason = null): void
    {
        $this->update([
            'escalated_to_human' => true,
            'assigned_agent_id' => $agentId,
            'escalated_at' => now(),
            'status' => 'escalated',
            'metadata' => array_merge($this->metadata ?? [], [
                'escalation_reason' => $reason,
            ]),
        ]);
    }

    /**
     * Mark conversation as resolved.
     */
    public function resolve(): void
    {
        $this->update([
            'status' => 'resolved',
            'resolved_at' => now(),
        ]);
    }

    /**
     * Close conversation.
     */
    public function close(): void
    {
        $this->update([
            'status' => 'closed',
            'closed_at' => now(),
        ]);
    }

    /**
     * Add satisfaction rating.
     */
    public function addRating(int $rating, ?string $feedback = null): void
    {
        $this->update([
            'satisfaction_rating' => $rating,
            'satisfaction_feedback' => $feedback,
        ]);
    }

    /**
     * Update average confidence score.
     */
    public function updateAverageConfidence(): void
    {
        $avgScore = $this->messages()
            ->whereNotNull('confidence_score')
            ->avg('confidence_score');

        $this->update([
            'avg_confidence_score' => $avgScore,
        ]);
    }

    /**
     * Increment message count.
     */
    public function incrementMessageCount(): void
    {
        $this->increment('total_messages');
    }

    /**
     * Get customer display name.
     */
    public function getCustomerDisplayNameAttribute(): string
    {
        return $this->customer_name ?? $this->customer?->name ?? 'Guest';
    }

    /**
     * Check if conversation is active.
     */
    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Check if conversation is escalated.
     */
    public function isEscalated(): bool
    {
        return $this->escalated_to_human;
    }
}