<?php

namespace App\Notifications;

use App\Models\SharedSocialAccess;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class SocialAccessGrantedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $sharedAccess;
    protected $recipientType;

    /**
     * Create a new notification instance.
     */
    public function __construct(SharedSocialAccess $sharedAccess, string $recipientType)
    {
        $this->sharedAccess = $sharedAccess;
        $this->recipientType = $recipientType; // 'creator' or 'brand'
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail($notifiable): MailMessage
    {
        $connection = $this->sharedAccess->connection;
        $brand = $this->sharedAccess->brand;
        $creator = $this->sharedAccess->creator;

        if ($this->recipientType === 'creator') {
            return (new MailMessage)
                ->subject('Social Account Access Granted')
                ->greeting("Hello {$creator->name}!")
                ->line("You have successfully granted access to your {$connection->platform_display_name} account.")
                ->line("**Brand:** {$brand->name}")
                ->line("**Access Level:** {$this->sharedAccess->access_level_display}")
                ->line("**Expires:** {$this->sharedAccess->expires_at->format('M d, Y')}")
                ->line("**Allowed Permissions:**")
                ->line(implode(', ', $this->sharedAccess->formatted_scopes))
                ->action('View Connection Details', route('creator.social-connections.show', $connection))
                ->line('You can revoke this access at any time from your social connections settings.')
                ->line('Thank you for using our platform!');
        } else {
            return (new MailMessage)
                ->subject('Creator Granted Social Account Access')
                ->greeting("Hello!")
                ->line("A creator has granted your brand access to their {$connection->platform_display_name} account.")
                ->line("**Creator:** {$creator->name}")
                ->line("**Platform:** {$connection->platform_display_name}")
                ->line("**Account:** @{$connection->platform_username}")
                ->line("**Access Level:** {$this->sharedAccess->access_level_display}")
                ->line("**Expires:** {$this->sharedAccess->expires_at->format('M d, Y')}")
                ->action('View Access Details', route('brand.creators.show', $creator))
                ->line('This access will automatically expire on the date shown above.')
                ->line('Thank you for using our platform!');
        }
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray($notifiable): array
    {
        $connection = $this->sharedAccess->connection;
        $brand = $this->sharedAccess->brand;
        $creator = $this->sharedAccess->creator;

        if ($this->recipientType === 'creator') {
            return [
                'type' => 'social_access_granted',
                'title' => 'Social Account Access Granted',
                'message' => "You granted {$brand->name} access to your {$connection->platform_display_name} account",
                'connection_id' => $connection->id,
                'brand_id' => $brand->id,
                'shared_access_id' => $this->sharedAccess->id,
                'platform' => $connection->platform,
                'expires_at' => $this->sharedAccess->expires_at->toIso8601String(),
                'action_url' => route('creator.social-connections.show', $connection),
            ];
        } else {
            return [
                'type' => 'social_access_received',
                'title' => 'Creator Granted Social Access',
                'message' => "{$creator->name} granted access to their {$connection->platform_display_name} account",
                'connection_id' => $connection->id,
                'creator_id' => $creator->id,
                'shared_access_id' => $this->sharedAccess->id,
                'platform' => $connection->platform,
                'expires_at' => $this->sharedAccess->expires_at->toIso8601String(),
                'action_url' => route('brand.creators.show', $creator),
            ];
        }
    }
}