<?php

namespace App\Policies;

use App\Models\Ad;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class AdPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any ads.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        // Only brand admins and superadmins can view ads
        return $user->hasRole('brand_admin') || $user->hasRole('superadmin');
    }

    /**
     * Determine whether the user can view the ad.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, Ad $ad)
    {
        // Superadmin can view all ads
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand admin can only view their own brand's ads
        return $user->brand && $user->brand->id === $ad->brand_id;
    }

    /**
     * Determine whether the user can create ads.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        // Only brand admins with an associated brand can create ads
        return $user->hasRole('brand_admin') && $user->brand !== null;
    }

    /**
     * Determine whether the user can update the ad.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Ad $ad)
    {
        // Superadmin can update all ads
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand admin can only update their own brand's ads
        return $user->brand && $user->brand->id === $ad->brand_id;
    }

    /**
     * Determine whether the user can delete the ad.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Ad $ad)
    {
        // Superadmin can delete all ads
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand admin can only delete their own brand's ads
        // Additionally, prevent deletion of active ads
        return $user->brand 
            && $user->brand->id === $ad->brand_id 
            && $ad->status !== 'active';
    }

    /**
     * Determine whether the user can restore the ad.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, Ad $ad)
    {
        // Superadmin can restore all ads
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand admin can only restore their own brand's ads
        return $user->brand && $user->brand->id === $ad->brand_id;
    }

    /**
     * Determine whether the user can permanently delete the ad.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, Ad $ad)
    {
        // Only superadmin can force delete ads
        return $user->hasRole('superadmin');
    }

    /**
     * Determine whether the user can manage ad variants.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function manageVariants(User $user, Ad $ad)
    {
        // Superadmin can manage all ad variants
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand admin can only manage their own brand's ad variants
        return $user->brand && $user->brand->id === $ad->brand_id;
    }

    /**
     * Determine whether the user can view A/B test results.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAbTestResults(User $user, Ad $ad)
    {
        // Superadmin can view all A/B test results
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand admin can only view their own brand's A/B test results
        return $user->brand && $user->brand->id === $ad->brand_id;
    }

    /**
     * Determine whether the user can export ad results.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function exportResults(User $user, Ad $ad)
    {
        // Superadmin can export all ad results
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand admin can only export their own brand's ad results
        return $user->brand && $user->brand->id === $ad->brand_id;
    }
}