<?php

namespace App\Policies;

use App\Models\FinancialReport;
use App\Models\User;

class FinancialReportPolicy
{
    /**
     * Determine if the user can view any financial reports
     */
    public function viewAny(User $user): bool
    {
        return $user->role === 'brand' || $user->role === 'admin';
    }

    /**
     * Determine if the user can view the financial report
     */
    public function view(User $user, FinancialReport $report): bool
    {
        // Admin can view all reports
        if ($user->role === 'admin') {
            return true;
        }

        // Brand can only view their own reports
        return $user->role === 'brand' && $user->id === $report->brand_id;
    }

    /**
     * Determine if the user can create financial reports
     */
    public function create(User $user): bool
    {
        return $user->role === 'brand' || $user->role === 'admin';
    }

    /**
     * Determine if the user can update the financial report
     */
    public function update(User $user, FinancialReport $report): bool
    {
        // Admin can update all reports
        if ($user->role === 'admin') {
            return true;
        }

        // Brand can only update their own reports
        return $user->role === 'brand' && $user->id === $report->brand_id;
    }

    /**
     * Determine if the user can delete the financial report
     */
    public function delete(User $user, FinancialReport $report): bool
    {
        // Admin can delete all reports
        if ($user->role === 'admin') {
            return true;
        }

        // Brand can only delete their own reports
        return $user->role === 'brand' && $user->id === $report->brand_id;
    }

    /**
     * Determine if the user can export the financial report
     */
    public function export(User $user, FinancialReport $report): bool
    {
        return $this->view($user, $report);
    }
}