<?php

namespace App\Policies;

use App\Models\PerformanceForecast;
use App\Models\User;

class PerformanceForecastPolicy
{
    /**
     * Determine if the user can view any forecasts.
     *
     * @param  \App\Models\User  $user
     * @return bool
     */
    public function viewAny(User $user)
    {
        // Only brand_admin role can view forecasts
        return $user->hasRole('brand_admin') && $user->brand_id !== null;
    }

    /**
     * Determine if the user can view the forecast.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return bool
     */
    public function view(User $user, PerformanceForecast $forecast)
    {
        // User must be brand_admin and forecast must belong to their brand
        return $user->hasRole('brand_admin') 
            && $user->brand_id === $forecast->brand_id;
    }

    /**
     * Determine if the user can create forecasts.
     *
     * @param  \App\Models\User  $user
     * @return bool
     */
    public function create(User $user)
    {
        // Only brand_admin role can create forecasts
        return $user->hasRole('brand_admin') && $user->brand_id !== null;
    }

    /**
     * Determine if the user can update the forecast.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return bool
     */
    public function update(User $user, PerformanceForecast $forecast)
    {
        // User must be brand_admin and forecast must belong to their brand
        // Only allow updates if forecast is not completed
        return $user->hasRole('brand_admin') 
            && $user->brand_id === $forecast->brand_id
            && !$forecast->isCompleted();
    }

    /**
     * Determine if the user can delete the forecast.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return bool
     */
    public function delete(User $user, PerformanceForecast $forecast)
    {
        // User must be brand_admin and forecast must belong to their brand
        return $user->hasRole('brand_admin') 
            && $user->brand_id === $forecast->brand_id;
    }

    /**
     * Determine if the user can export the forecast.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PerformanceForecast  $forecast
     * @return bool
     */
    public function export(User $user, PerformanceForecast $forecast)
    {
        // User must be brand_admin, forecast must belong to their brand, and be completed
        return $user->hasRole('brand_admin') 
            && $user->brand_id === $forecast->brand_id
            && $forecast->isCompleted();
    }

    /**
     * Determine if the user can compare scenarios.
     *
     * @param  \App\Models\User  $user
     * @return bool
     */
    public function compareScenarios(User $user)
    {
        // Only brand_admin role can compare scenarios
        return $user->hasRole('brand_admin') && $user->brand_id !== null;
    }
}