<?php

namespace App\Policies;

use App\Models\ProviderCredential;
use App\Models\User;

class ProviderCredentialPolicy
{
    /**
     * Determine if the user can view any provider credentials.
     */
    public function viewAny(User $user): bool
    {
        return $user->hasRole(['brand_admin', 'superadmin']);
    }

    /**
     * Determine if the user can view the provider credential.
     */
    public function view(User $user, ProviderCredential $credential): bool
    {
        // SuperAdmin can view all
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand Admin can only view their own brand's credentials
        return $user->hasRole('brand_admin') && 
               $user->brand_id === $credential->brand_id;
    }

    /**
     * Determine if the user can create provider credentials.
     */
    public function create(User $user): bool
    {
        return $user->hasRole(['brand_admin', 'superadmin']) && $user->brand_id !== null;
    }

    /**
     * Determine if the user can update the provider credential.
     */
    public function update(User $user, ProviderCredential $credential): bool
    {
        // SuperAdmin can update all
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand Admin can only update their own brand's credentials
        return $user->hasRole('brand_admin') && 
               $user->brand_id === $credential->brand_id;
    }

    /**
     * Determine if the user can delete the provider credential.
     */
    public function delete(User $user, ProviderCredential $credential): bool
    {
        // SuperAdmin can delete all
        if ($user->hasRole('superadmin')) {
            return true;
        }

        // Brand Admin can only delete their own brand's credentials
        return $user->hasRole('brand_admin') && 
               $user->brand_id === $credential->brand_id;
    }
}