<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Brand;
use App\Models\SupportConversation;
use App\Models\KnowledgeBaseDocument;

class SupportPolicy
{
    /**
     * Determine if user can manage support for a brand.
     */
    public function manageSupport(User $user, Brand $brand): bool
    {
        // Brand owner or admin
        if ($user->brand_id === $brand->id) {
            return $user->hasRole(['brand_admin', 'brand_owner']);
        }

        // Agency managing the brand
        if ($user->hasRole('agency')) {
            return $user->brands()->where('brands.id', $brand->id)->exists();
        }

        // Super admin
        return $user->hasRole('super_admin');
    }

    /**
     * Determine if user can view conversations.
     */
    public function viewConversations(User $user, Brand $brand): bool
    {
        return $this->manageSupport($user, $brand);
    }

    /**
     * Determine if user can manage knowledge base.
     */
    public function manageKnowledgeBase(User $user, Brand $brand): bool
    {
        return $this->manageSupport($user, $brand);
    }

    /**
     * Determine if user can view specific conversation.
     */
    public function viewConversation(User $user, SupportConversation $conversation): bool
    {
        // Customer can view their own conversation
        if ($conversation->customer_id === $user->id) {
            return true;
        }

        // Assigned agent can view
        if ($conversation->assigned_agent_id === $user->id) {
            return true;
        }

        // Brand admin can view
        return $this->manageSupport($user, $conversation->brand);
    }

    /**
     * Determine if user can respond to conversation.
     */
    public function respondToConversation(User $user, SupportConversation $conversation): bool
    {
        // Assigned agent can respond
        if ($conversation->assigned_agent_id === $user->id) {
            return true;
        }

        // Brand admin can respond
        return $this->manageSupport($user, $conversation->brand);
    }

    /**
     * Determine if user can view knowledge base document.
     */
    public function viewDocument(User $user, KnowledgeBaseDocument $document): bool
    {
        return $this->manageSupport($user, $document->brand);
    }

    /**
     * Determine if user can create knowledge base document.
     */
    public function createDocument(User $user, Brand $brand): bool
    {
        return $this->manageSupport($user, $brand);
    }

    /**
     * Determine if user can update knowledge base document.
     */
    public function updateDocument(User $user, KnowledgeBaseDocument $document): bool
    {
        return $this->manageSupport($user, $document->brand);
    }

    /**
     * Determine if user can delete knowledge base document.
     */
    public function deleteDocument(User $user, KnowledgeBaseDocument $document): bool
    {
        return $this->manageSupport($user, $document->brand);
    }

    /**
     * Determine if user can update bot settings.
     */
    public function updateBotSettings(User $user, Brand $brand): bool
    {
        // Only brand owner or super admin
        if ($user->brand_id === $brand->id) {
            return $user->hasRole(['brand_owner', 'brand_admin']);
        }

        return $user->hasRole('super_admin');
    }

    /**
     * Determine if user can view analytics.
     */
    public function viewAnalytics(User $user, Brand $brand): bool
    {
        return $this->manageSupport($user, $brand);
    }
}