<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class AITranslationService
{
    protected string $apiKey;
    protected string $apiUrl;
    protected string $model;

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key');
        $this->apiUrl = config('services.openai.api_url', 'https://api.openai.com/v1/chat/completions');
        $this->model = config('services.openai.model', 'gpt-4');
    }

    /**
     * Translate text to target locale with cultural adaptation.
     *
     * @param string $sourceText The text to translate
     * @param string $sourceLocale Source language locale (e.g., 'en-US')
     * @param string $targetLocale Target language locale (e.g., 'es-ES')
     * @param string $tone Tone of translation: 'formal', 'friendly', 'neutral'
     * @param string|null $context Additional context about the content
     * @return array ['translated_text' => string, 'cultural_notes' => string, 'job_id' => string]
     * @throws Exception
     */
    public function translate(
        string $sourceText,
        string $sourceLocale,
        string $targetLocale,
        string $tone = 'neutral',
        ?string $context = null
    ): array {
        try {
            $jobId = $this->generateJobId();
            
            $prompt = $this->buildTranslationPrompt(
                $sourceText,
                $sourceLocale,
                $targetLocale,
                $tone,
                $context
            );

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(60)->post($this->apiUrl, [
                'model' => $this->model,
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are an expert translator and cultural consultant specializing in marketing and advertising content localization. You provide accurate translations that respect cultural nuances and local preferences.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.3,
                'max_tokens' => 2000,
            ]);

            if (!$response->successful()) {
                throw new Exception('AI translation API request failed: ' . $response->body());
            }

            $result = $response->json();
            $content = $result['choices'][0]['message']['content'] ?? '';

            // Parse the AI response
            $parsed = $this->parseAIResponse($content);

            Log::info('AI Translation completed', [
                'job_id' => $jobId,
                'source_locale' => $sourceLocale,
                'target_locale' => $targetLocale,
                'tone' => $tone,
            ]);

            return [
                'translated_text' => $parsed['translated_text'],
                'cultural_notes' => $parsed['cultural_notes'],
                'job_id' => $jobId,
            ];

        } catch (Exception $e) {
            Log::error('AI Translation failed', [
                'error' => $e->getMessage(),
                'source_locale' => $sourceLocale,
                'target_locale' => $targetLocale,
            ]);

            throw $e;
        }
    }

    /**
     * Translate multiple texts in batch.
     *
     * @param array $texts Array of texts to translate
     * @param string $sourceLocale Source language locale
     * @param string $targetLocale Target language locale
     * @param string $tone Tone of translation
     * @return array Array of translation results
     */
    public function batchTranslate(
        array $texts,
        string $sourceLocale,
        string $targetLocale,
        string $tone = 'neutral'
    ): array {
        $results = [];

        foreach ($texts as $key => $text) {
            try {
                $results[$key] = $this->translate($text, $sourceLocale, $targetLocale, $tone);
            } catch (Exception $e) {
                $results[$key] = [
                    'error' => $e->getMessage(),
                    'translated_text' => null,
                    'cultural_notes' => null,
                    'job_id' => null,
                ];
            }
        }

        return $results;
    }

    /**
     * Build the translation prompt for AI.
     */
    protected function buildTranslationPrompt(
        string $sourceText,
        string $sourceLocale,
        string $targetLocale,
        string $tone,
        ?string $context
    ): string {
        $toneInstructions = [
            'formal' => 'Use formal language, professional terminology, and respectful address forms.',
            'friendly' => 'Use casual, warm language that feels approachable and conversational.',
            'neutral' => 'Use balanced language that is neither too formal nor too casual.',
        ];

        $toneInstruction = $toneInstructions[$tone] ?? $toneInstructions['neutral'];

        $prompt = "Translate the following marketing/advertising content from {$sourceLocale} to {$targetLocale}.\n\n";
        $prompt .= "TONE: {$tone} - {$toneInstruction}\n\n";
        
        if ($context) {
            $prompt .= "CONTEXT: {$context}\n\n";
        }

        $prompt .= "SOURCE TEXT:\n{$sourceText}\n\n";
        $prompt .= "Please provide:\n";
        $prompt .= "1. TRANSLATION: The translated text that maintains the original meaning while adapting to local cultural context\n";
        $prompt .= "2. CULTURAL NOTES: Important cultural considerations, local preferences, or adaptations made (2-3 sentences)\n\n";
        $prompt .= "Format your response as:\n";
        $prompt .= "TRANSLATION:\n[your translation here]\n\n";
        $prompt .= "CULTURAL NOTES:\n[your cultural notes here]";

        return $prompt;
    }

    /**
     * Parse AI response to extract translation and cultural notes.
     */
    protected function parseAIResponse(string $content): array
    {
        $translatedText = '';
        $culturalNotes = '';

        // Extract translation
        if (preg_match('/TRANSLATION:\s*\n(.*?)\n\nCULTURAL NOTES:/s', $content, $matches)) {
            $translatedText = trim($matches[1]);
        } elseif (preg_match('/TRANSLATION:\s*\n(.*?)$/s', $content, $matches)) {
            $translatedText = trim($matches[1]);
        }

        // Extract cultural notes
        if (preg_match('/CULTURAL NOTES:\s*\n(.*)$/s', $content, $matches)) {
            $culturalNotes = trim($matches[1]);
        }

        // Fallback: if parsing fails, use the entire content as translation
        if (empty($translatedText)) {
            $translatedText = trim($content);
            $culturalNotes = 'No specific cultural notes provided.';
        }

        return [
            'translated_text' => $translatedText,
            'cultural_notes' => $culturalNotes,
        ];
    }

    /**
     * Generate a unique job ID for tracking.
     */
    protected function generateJobId(): string
    {
        return 'ai_trans_' . uniqid() . '_' . time();
    }

    /**
     * Get supported locales.
     */
    public static function getSupportedLocales(): array
    {
        return [
            'en' => 'English',
            'en-US' => 'English (US)',
            'en-GB' => 'English (UK)',
            'es' => 'Spanish',
            'es-ES' => 'Spanish (Spain)',
            'es-MX' => 'Spanish (Mexico)',
            'fr' => 'French',
            'fr-FR' => 'French (France)',
            'de' => 'German',
            'de-DE' => 'German (Germany)',
            'it' => 'Italian',
            'it-IT' => 'Italian (Italy)',
            'pt' => 'Portuguese',
            'pt-BR' => 'Portuguese (Brazil)',
            'pt-PT' => 'Portuguese (Portugal)',
            'ja' => 'Japanese',
            'ja-JP' => 'Japanese (Japan)',
            'zh' => 'Chinese',
            'zh-CN' => 'Chinese (Simplified)',
            'zh-TW' => 'Chinese (Traditional)',
            'ko' => 'Korean',
            'ko-KR' => 'Korean (Korea)',
            'ar' => 'Arabic',
            'ar-SA' => 'Arabic (Saudi Arabia)',
            'hi' => 'Hindi',
            'hi-IN' => 'Hindi (India)',
        ];
    }

    /**
     * Get supported tones.
     */
    public static function getSupportedTones(): array
    {
        return [
            'formal' => 'Formal - Professional and respectful',
            'friendly' => 'Friendly - Casual and approachable',
            'neutral' => 'Neutral - Balanced tone',
        ];
    }
}