<?php

namespace App\Services;

use App\Models\CreatorSocialConnection;
use App\Models\SharedSocialAccess;
use App\Models\AuditLog;
use App\Models\Brand;
use App\Notifications\SocialAccessGrantedNotification;
use App\Notifications\SocialAccessRevokedNotification;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Exception;

class SocialConnectionService
{
    /**
     * Test social connection with read-only API call.
     */
    public function testConnection(CreatorSocialConnection $connection): array
    {
        try {
            $result = match($connection->platform) {
                'youtube' => $this->testYouTube($connection),
                'instagram' => $this->testInstagram($connection),
                'tiktok' => $this->testTikTok($connection),
                'linkedin' => $this->testLinkedIn($connection),
                'twitter' => $this->testTwitter($connection),
                'facebook' => $this->testFacebook($connection),
                default => throw new Exception("Unsupported platform: {$connection->platform}"),
            };

            // Update connection status and profile info
            $connection->update([
                'status' => 'connected',
                'last_tested_at' => now(),
                'last_synced_at' => now(),
                'last_error' => null,
                'platform_user_id' => $result['user_id'] ?? $connection->platform_user_id,
                'platform_username' => $result['username'] ?? $connection->platform_username,
                'platform_display_name' => $result['display_name'] ?? $connection->platform_display_name,
                'profile_picture_url' => $result['profile_picture'] ?? $connection->profile_picture_url,
            ]);

            // Log successful test
            $this->logAudit($connection, 'test_connection_success', $result);

            return [
                'success' => true,
                'message' => 'Connection successful',
                'data' => $result,
            ];

        } catch (Exception $e) {
            // Update connection with error
            $connection->update([
                'status' => 'failed',
                'last_tested_at' => now(),
                'last_error' => $e->getMessage(),
            ]);

            // Log failed test
            $this->logAudit($connection, 'test_connection_failed', [
                'error' => $e->getMessage(),
            ]);

            Log::error("Social connection test failed", [
                'platform' => $connection->platform,
                'connection_id' => $connection->id,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $this->getUserFriendlyError($e->getMessage()),
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Test YouTube connection.
     */
    protected function testYouTube(CreatorSocialConnection $connection): array
    {
        $response = Http::withToken($connection->access_token)
            ->get('https://www.googleapis.com/youtube/v3/channels', [
                'part' => 'snippet,statistics',
                'mine' => 'true',
            ]);

        if (!$response->successful()) {
            throw new Exception("YouTube API error: " . $response->body());
        }

        $data = $response->json();
        $channel = $data['items'][0] ?? null;

        if (!$channel) {
            throw new Exception("No YouTube channel found");
        }

        return [
            'user_id' => $channel['id'],
            'username' => $channel['snippet']['customUrl'] ?? null,
            'display_name' => $channel['snippet']['title'],
            'profile_picture' => $channel['snippet']['thumbnails']['default']['url'] ?? null,
            'subscriber_count' => $channel['statistics']['subscriberCount'] ?? 0,
        ];
    }

    /**
     * Test Instagram connection.
     */
    protected function testInstagram(CreatorSocialConnection $connection): array
    {
        $response = Http::get('https://graph.instagram.com/me', [
            'fields' => 'id,username,account_type,media_count',
            'access_token' => $connection->access_token,
        ]);

        if (!$response->successful()) {
            throw new Exception("Instagram API error: " . $response->body());
        }

        $data = $response->json();

        return [
            'user_id' => $data['id'],
            'username' => $data['username'],
            'display_name' => $data['username'],
            'account_type' => $data['account_type'] ?? 'PERSONAL',
            'media_count' => $data['media_count'] ?? 0,
        ];
    }

    /**
     * Test TikTok connection.
     */
    protected function testTikTok(CreatorSocialConnection $connection): array
    {
        $response = Http::withToken($connection->access_token)
            ->get('https://open-api.tiktok.com/user/info/');

        if (!$response->successful()) {
            throw new Exception("TikTok API error: " . $response->body());
        }

        $data = $response->json();
        $user = $data['data']['user'] ?? null;

        if (!$user) {
            throw new Exception("No TikTok user found");
        }

        return [
            'user_id' => $user['open_id'],
            'username' => $user['unique_id'] ?? null,
            'display_name' => $user['display_name'],
            'profile_picture' => $user['avatar_url'] ?? null,
        ];
    }

    /**
     * Test LinkedIn connection.
     */
    protected function testLinkedIn(CreatorSocialConnection $connection): array
    {
        $response = Http::withToken($connection->access_token)
            ->get('https://api.linkedin.com/v2/me');

        if (!$response->successful()) {
            throw new Exception("LinkedIn API error: " . $response->body());
        }

        $data = $response->json();

        return [
            'user_id' => $data['id'],
            'username' => null,
            'display_name' => ($data['localizedFirstName'] ?? '') . ' ' . ($data['localizedLastName'] ?? ''),
            'profile_picture' => $data['profilePicture']['displayImage~']['elements'][0]['identifiers'][0]['identifier'] ?? null,
        ];
    }

    /**
     * Test Twitter/X connection.
     */
    protected function testTwitter(CreatorSocialConnection $connection): array
    {
        $response = Http::withToken($connection->access_token)
            ->get('https://api.twitter.com/2/users/me', [
                'user.fields' => 'profile_image_url,public_metrics',
            ]);

        if (!$response->successful()) {
            throw new Exception("Twitter API error: " . $response->body());
        }

        $data = $response->json();
        $user = $data['data'] ?? null;

        if (!$user) {
            throw new Exception("No Twitter user found");
        }

        return [
            'user_id' => $user['id'],
            'username' => $user['username'],
            'display_name' => $user['name'],
            'profile_picture' => $user['profile_image_url'] ?? null,
            'followers_count' => $user['public_metrics']['followers_count'] ?? 0,
        ];
    }

    /**
     * Test Facebook connection.
     */
    protected function testFacebook(CreatorSocialConnection $connection): array
    {
        $response = Http::get('https://graph.facebook.com/me', [
            'fields' => 'id,name,picture',
            'access_token' => $connection->access_token,
        ]);

        if (!$response->successful()) {
            throw new Exception("Facebook API error: " . $response->body());
        }

        $data = $response->json();

        return [
            'user_id' => $data['id'],
            'username' => null,
            'display_name' => $data['name'],
            'profile_picture' => $data['picture']['data']['url'] ?? null,
        ];
    }

    /**
     * Share connection access with a brand.
     */
    public function shareWithBrand(
        CreatorSocialConnection $connection,
        Brand $brand,
        array $scopes,
        string $accessLevel,
        int $expiryDays,
        ?int $campaignId = null
    ): SharedSocialAccess {
        // Validate connection can be shared
        if (!$connection->canBeShared()) {
            throw new Exception("Connection cannot be shared in its current state");
        }

        // Generate scoped access token (simplified - in production, use proper OAuth scoping)
        $scopedToken = $this->generateScopedToken($connection, $scopes);

        // Create shared access record
        $sharedAccess = SharedSocialAccess::create([
            'connection_id' => $connection->id,
            'creator_id' => $connection->creator_id,
            'brand_id' => $brand->id,
            'campaign_id' => $campaignId,
            'allowed_scopes' => $scopes,
            'access_level' => $accessLevel,
            'scoped_access_token' => $scopedToken,
            'token_expires_at' => $connection->token_expires_at,
            'granted_at' => now(),
            'expires_at' => now()->addDays($expiryDays),
            'status' => 'active',
            'creator_consent' => true,
            'consent_given_at' => now(),
            'consent_ip' => request()->ip(),
            'consent_user_agent' => request()->userAgent(),
            'granted_by' => auth()->id(),
        ]);

        // Send notifications
        $connection->creator->notify(new SocialAccessGrantedNotification($sharedAccess, 'creator'));
        
        // Notify brand admins
        $brand->users()->where('role', 'brand_admin')->each(function ($user) use ($sharedAccess) {
            $user->notify(new SocialAccessGrantedNotification($sharedAccess, 'brand'));
        });

        $sharedAccess->update([
            'creator_notified' => true,
            'brand_notified' => true,
        ]);

        // Log the sharing
        $this->logAudit($connection, 'access_shared', [
            'brand_id' => $brand->id,
            'brand_name' => $brand->name,
            'scopes' => $scopes,
            'access_level' => $accessLevel,
            'expires_at' => $sharedAccess->expires_at->toIso8601String(),
        ]);

        return $sharedAccess;
    }

    /**
     * Revoke shared access.
     */
    public function revokeSharedAccess(SharedSocialAccess $sharedAccess, string $reason = null): bool
    {
        $sharedAccess->revoke($reason);

        // Send notifications
        $sharedAccess->creator->notify(new SocialAccessRevokedNotification($sharedAccess, 'creator'));
        
        $sharedAccess->brand->users()->where('role', 'brand_admin')->each(function ($user) use ($sharedAccess) {
            $user->notify(new SocialAccessRevokedNotification($sharedAccess, 'brand'));
        });

        // Log the revocation
        $this->logAudit($sharedAccess->connection, 'access_revoked', [
            'brand_id' => $sharedAccess->brand_id,
            'brand_name' => $sharedAccess->brand->name,
            'reason' => $reason,
        ]);

        return true;
    }

    /**
     * Generate scoped access token.
     * In production, this should use proper OAuth token scoping.
     */
    protected function generateScopedToken(CreatorSocialConnection $connection, array $scopes): string
    {
        // This is a simplified implementation
        // In production, you would:
        // 1. Use the platform's OAuth API to generate a scoped token
        // 2. Or implement your own token proxy service
        
        return base64_encode(json_encode([
            'connection_id' => $connection->id,
            'scopes' => $scopes,
            'token' => Str::random(64),
            'created_at' => now()->timestamp,
        ]));
    }

    /**
     * Refresh OAuth token.
     */
    public function refreshToken(CreatorSocialConnection $connection): bool
    {
        if (empty($connection->refresh_token)) {
            return false;
        }

        try {
            $result = match($connection->platform) {
                'youtube' => $this->refreshGoogleToken($connection),
                'instagram', 'facebook' => $this->refreshFacebookToken($connection),
                'linkedin' => $this->refreshLinkedInToken($connection),
                'twitter' => $this->refreshTwitterToken($connection),
                default => false,
            };

            if ($result) {
                $this->logAudit($connection, 'token_refreshed');
            }

            return $result;

        } catch (Exception $e) {
            Log::error("Token refresh failed", [
                'platform' => $connection->platform,
                'connection_id' => $connection->id,
                'error' => $e->getMessage(),
            ]);

            return false;
        }
    }

    /**
     * Refresh Google OAuth token.
     */
    protected function refreshGoogleToken(CreatorSocialConnection $connection): bool
    {
        $response = Http::asForm()->post('https://oauth2.googleapis.com/token', [
            'client_id' => $connection->client_id ?? config('services.google.client_id'),
            'client_secret' => $connection->client_secret ?? config('services.google.client_secret'),
            'refresh_token' => $connection->refresh_token,
            'grant_type' => 'refresh_token',
        ]);

        if (!$response->successful()) {
            return false;
        }

        $data = $response->json();
        
        $connection->update([
            'access_token' => $data['access_token'],
            'token_expires_at' => now()->addSeconds($data['expires_in'] ?? 3600),
            'expiry_alert_sent' => false,
        ]);

        return true;
    }

    /**
     * Refresh Facebook OAuth token.
     */
    protected function refreshFacebookToken(CreatorSocialConnection $connection): bool
    {
        $response = Http::get('https://graph.facebook.com/v18.0/oauth/access_token', [
            'grant_type' => 'fb_exchange_token',
            'client_id' => $connection->client_id ?? config('services.facebook.client_id'),
            'client_secret' => $connection->client_secret ?? config('services.facebook.client_secret'),
            'fb_exchange_token' => $connection->access_token,
        ]);

        if (!$response->successful()) {
            return false;
        }

        $data = $response->json();
        
        $connection->update([
            'access_token' => $data['access_token'],
            'token_expires_at' => now()->addSeconds($data['expires_in'] ?? 5184000),
            'expiry_alert_sent' => false,
        ]);

        return true;
    }

    /**
     * Refresh LinkedIn OAuth token.
     */
    protected function refreshLinkedInToken(CreatorSocialConnection $connection): bool
    {
        $response = Http::asForm()->post('https://www.linkedin.com/oauth/v2/accessToken', [
            'grant_type' => 'refresh_token',
            'refresh_token' => $connection->refresh_token,
            'client_id' => $connection->client_id ?? config('services.linkedin.client_id'),
            'client_secret' => $connection->client_secret ?? config('services.linkedin.client_secret'),
        ]);

        if (!$response->successful()) {
            return false;
        }

        $data = $response->json();
        
        $connection->update([
            'access_token' => $data['access_token'],
            'refresh_token' => $data['refresh_token'] ?? $connection->refresh_token,
            'token_expires_at' => now()->addSeconds($data['expires_in'] ?? 5184000),
            'expiry_alert_sent' => false,
        ]);

        return true;
    }

    /**
     * Refresh Twitter OAuth token.
     */
    protected function refreshTwitterToken(CreatorSocialConnection $connection): bool
    {
        $response = Http::asForm()
            ->withBasicAuth(
                $connection->client_id ?? config('services.twitter.client_id'),
                $connection->client_secret ?? config('services.twitter.client_secret')
            )
            ->post('https://api.twitter.com/2/oauth2/token', [
                'grant_type' => 'refresh_token',
                'refresh_token' => $connection->refresh_token,
            ]);

        if (!$response->successful()) {
            return false;
        }

        $data = $response->json();
        
        $connection->update([
            'access_token' => $data['access_token'],
            'refresh_token' => $data['refresh_token'] ?? $connection->refresh_token,
            'token_expires_at' => now()->addSeconds($data['expires_in'] ?? 7200),
            'expiry_alert_sent' => false,
        ]);

        return true;
    }

    /**
     * Log audit trail.
     */
    protected function logAudit(CreatorSocialConnection $connection, string $action, array $meta = []): void
    {
        AuditLog::create([
            'actor_id' => auth()->id(),
            'action' => $action,
            'target_type' => CreatorSocialConnection::class,
            'target_id' => $connection->id,
            'meta' => array_merge([
                'platform' => $connection->platform,
                'label' => $connection->label,
                'creator_id' => $connection->creator_id,
            ], $meta),
        ]);
    }

    /**
     * Get user-friendly error message.
     */
    protected function getUserFriendlyError(string $error): string
    {
        if (str_contains($error, 'invalid_grant')) {
            return 'Your access token has expired or been revoked. Please reconnect your account.';
        }

        if (str_contains($error, 'invalid_client')) {
            return 'Invalid credentials. Please reconnect your account.';
        }

        if (str_contains($error, 'insufficient_scope')) {
            return 'Insufficient permissions. Please reconnect with the required permissions.';
        }

        if (str_contains($error, 'rate_limit')) {
            return 'Rate limit exceeded. Please try again later.';
        }

        if (str_contains($error, 'unauthorized')) {
            return 'Authentication failed. Please check your connection and try again.';
        }

        return 'Connection test failed. Please verify your connection and try again.';
    }
}