<?php

namespace Database\Factories;

use App\Models\Campaign;
use App\Models\CampaignFinancial;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;

class CampaignFinancialFactory extends Factory
{
    protected $model = CampaignFinancial::class;

    public function definition(): array
    {
        $adSpend = $this->faker->randomFloat(2, 1000, 50000);
        $influencerPayments = $this->faker->randomFloat(2, 500, 20000);
        $productionCosts = $this->faker->randomFloat(2, 200, 5000);
        $otherCosts = $this->faker->randomFloat(2, 0, 2000);
        
        $clicks = $this->faker->numberBetween(1000, 100000);
        $impressions = $clicks * $this->faker->numberBetween(10, 50);
        $conversions = (int) ($clicks * $this->faker->randomFloat(2, 0.01, 0.05));
        $avgOrderValue = $this->faker->randomFloat(2, 20, 200);

        $periodStart = $this->faker->dateTimeBetween('-60 days', '-30 days');
        $periodEnd = (clone $periodStart)->modify('+' . $this->faker->numberBetween(7, 30) . ' days');

        return [
            'campaign_id' => Campaign::factory(),
            'brand_id' => User::factory()->brand(),
            'ad_spend' => $adSpend,
            'influencer_payments' => $influencerPayments,
            'production_costs' => $productionCosts,
            'other_costs' => $otherCosts,
            'clicks' => $clicks,
            'impressions' => $impressions,
            'conversions' => $conversions,
            'avg_order_value' => $avgOrderValue,
            'period_start' => $periodStart,
            'period_end' => $periodEnd,
            'spend_breakdown' => [
                'instagram' => $this->faker->randomFloat(2, 0, $adSpend * 0.4),
                'facebook' => $this->faker->randomFloat(2, 0, $adSpend * 0.3),
                'tiktok' => $this->faker->randomFloat(2, 0, $adSpend * 0.3),
            ],
            'performance_metrics' => [
                'ctr' => $this->faker->randomFloat(2, 1, 5),
                'engagement_rate' => $this->faker->randomFloat(2, 2, 10),
                'revenue_by_channel' => [
                    'instagram' => $this->faker->randomFloat(2, 1000, 20000),
                    'facebook' => $this->faker->randomFloat(2, 1000, 15000),
                    'tiktok' => $this->faker->randomFloat(2, 1000, 18000),
                ],
            ],
        ];
    }

    /**
     * Indicate that the campaign is highly profitable
     */
    public function profitable(): static
    {
        return $this->state(function (array $attributes) {
            $totalSpend = $attributes['ad_spend'] + $attributes['influencer_payments'] + 
                         $attributes['production_costs'] + $attributes['other_costs'];
            
            // Generate revenue that's 150-200% of spend
            $conversions = $attributes['conversions'];
            $avgOrderValue = ($totalSpend * $this->faker->randomFloat(2, 1.5, 2.0)) / $conversions;

            return [
                'avg_order_value' => $avgOrderValue,
            ];
        });
    }

    /**
     * Indicate that the campaign is breaking even
     */
    public function breakeven(): static
    {
        return $this->state(function (array $attributes) {
            $totalSpend = $attributes['ad_spend'] + $attributes['influencer_payments'] + 
                         $attributes['production_costs'] + $attributes['other_costs'];
            
            // Generate revenue that's 95-105% of spend
            $conversions = $attributes['conversions'];
            $avgOrderValue = ($totalSpend * $this->faker->randomFloat(2, 0.95, 1.05)) / $conversions;

            return [
                'avg_order_value' => $avgOrderValue,
            ];
        });
    }

    /**
     * Indicate that the campaign is losing money
     */
    public function unprofitable(): static
    {
        return $this->state(function (array $attributes) {
            $totalSpend = $attributes['ad_spend'] + $attributes['influencer_payments'] + 
                         $attributes['production_costs'] + $attributes['other_costs'];
            
            // Generate revenue that's 50-80% of spend
            $conversions = $attributes['conversions'];
            $avgOrderValue = ($totalSpend * $this->faker->randomFloat(2, 0.5, 0.8)) / $conversions;

            return [
                'avg_order_value' => $avgOrderValue,
            ];
        });
    }
}