<?php

namespace Database\Factories;

use App\Models\Brand;
use App\Models\PerformanceForecast;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\PerformanceForecast>
 */
class PerformanceForecastFactory extends Factory
{
    /**
     * The name of the factory's corresponding model.
     *
     * @var string
     */
    protected $model = PerformanceForecast::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'brand_id' => Brand::factory(),
            'created_by' => User::factory(),
            'campaign_ids' => null,
            'social_account_ids' => null,
            'start_date' => now()->subDays(60),
            'end_date' => now(),
            'forecast_days' => 30,
            'input_metrics' => [
                'impressions' => [1000, 1100, 1200],
                'clicks' => [50, 55, 60],
                'conversions' => [5, 6, 7],
                'engagement' => [100, 110, 120],
                'reach' => [800, 900, 1000],
                'dates' => [
                    now()->subDays(2)->format('Y-m-d'),
                    now()->subDays(1)->format('Y-m-d'),
                    now()->format('Y-m-d'),
                ],
            ],
            'scenario' => 'balanced',
            'predicted_timeseries' => [],
            'error_bands' => [],
            'roi_estimates' => [],
            'insights_text' => null,
            'insights_meta' => null,
            'status' => 'pending',
            'error_message' => null,
            'completed_at' => null,
        ];
    }

    /**
     * Indicate that the forecast is completed.
     *
     * @return static
     */
    public function completed()
    {
        return $this->state(function (array $attributes) {
            return [
                'status' => 'completed',
                'completed_at' => now(),
                'predicted_timeseries' => [
                    'dates' => [
                        now()->addDays(1)->format('Y-m-d'),
                        now()->addDays(2)->format('Y-m-d'),
                        now()->addDays(3)->format('Y-m-d'),
                    ],
                    'impressions' => [1300, 1400, 1500],
                    'clicks' => [65, 70, 75],
                    'conversions' => [8, 9, 10],
                    'engagement' => [130, 140, 150],
                    'reach' => [1100, 1200, 1300],
                ],
                'error_bands' => [
                    'impressions' => [
                        'lower' => [1200, 1300, 1400],
                        'upper' => [1400, 1500, 1600],
                    ],
                    'clicks' => [
                        'lower' => [60, 65, 70],
                        'upper' => [70, 75, 80],
                    ],
                    'conversions' => [
                        'lower' => [7, 8, 9],
                        'upper' => [9, 10, 11],
                    ],
                    'engagement' => [
                        'lower' => [120, 130, 140],
                        'upper' => [140, 150, 160],
                    ],
                    'reach' => [
                        'lower' => [1000, 1100, 1200],
                        'upper' => [1200, 1300, 1400],
                    ],
                ],
                'roi_estimates' => [
                    'roi' => 15.5,
                    'reach' => 50000,
                    'conversions' => 500,
                    'engagement_rate' => 3.2,
                    'ctr' => 2.5,
                    'conversion_rate' => 1.8,
                    'estimated_revenue' => 25000,
                    'total_impressions' => 100000,
                    'total_clicks' => 2500,
                    'total_engagement' => 3200,
                ],
                'insights_text' => 'Expected ROI of 15.5% indicates strong campaign performance. Shift budget to reels for +10% engagement boost.',
                'insights_meta' => [
                    'actionable_insights' => [
                        [
                            'type' => 'positive',
                            'title' => 'Positive ROI Expected',
                            'description' => 'Forecast shows a 15.5% ROI over the next 30 days.',
                            'action' => 'Continue current strategy',
                        ],
                        [
                            'type' => 'recommendation',
                            'title' => 'Boost Engagement',
                            'description' => 'Engagement rate of 3.2% is below industry average.',
                            'action' => 'Shift budget to reels and interactive content for +10% engagement boost',
                        ],
                    ],
                    'scenario' => 'balanced',
                    'confidence_level' => 0.95,
                ],
            ];
        });
    }

    /**
     * Indicate that the forecast is processing.
     *
     * @return static
     */
    public function processing()
    {
        return $this->state(function (array $attributes) {
            return [
                'status' => 'processing',
            ];
        });
    }

    /**
     * Indicate that the forecast has failed.
     *
     * @return static
     */
    public function failed()
    {
        return $this->state(function (array $attributes) {
            return [
                'status' => 'failed',
                'error_message' => 'Failed to process forecast: Insufficient historical data',
            ];
        });
    }

    /**
     * Indicate that the forecast uses conservative scenario.
     *
     * @return static
     */
    public function conservative()
    {
        return $this->state(function (array $attributes) {
            return [
                'scenario' => 'conservative',
            ];
        });
    }

    /**
     * Indicate that the forecast uses aggressive scenario.
     *
     * @return static
     */
    public function aggressive()
    {
        return $this->state(function (array $attributes) {
            return [
                'scenario' => 'aggressive',
            ];
        });
    }
}