<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('creator_social_connections', function (Blueprint $table) {
            $table->id();
            $table->foreignId('creator_id')->constrained('users')->onDelete('cascade');
            $table->string('platform'); // instagram, youtube, twitter, tiktok, linkedin
            $table->string('label')->nullable(); // e.g., "Personal Instagram", "Business YouTube"
            
            // OAuth credentials (encrypted)
            $table->text('client_id')->nullable();
            $table->text('client_secret')->nullable();
            $table->text('access_token')->nullable();
            $table->text('refresh_token')->nullable();
            
            // Platform-specific fields
            $table->text('ad_account_id')->nullable(); // For ad platforms
            $table->json('scopes')->nullable();
            $table->string('redirect_uri')->nullable();
            
            // Profile information (cached from platform)
            $table->string('platform_user_id')->nullable();
            $table->string('platform_username')->nullable();
            $table->string('platform_display_name')->nullable();
            $table->string('profile_picture_url')->nullable();
            
            // Token expiry tracking
            $table->timestamp('token_expires_at')->nullable();
            $table->boolean('expiry_alert_sent')->default(false);
            
            // Connection status
            $table->enum('status', ['connected', 'expiring', 'failed', 'disconnected'])->default('disconnected');
            $table->timestamp('last_tested_at')->nullable();
            $table->timestamp('last_synced_at')->nullable();
            $table->text('last_error')->nullable();
            
            // Ad usage flag (requires explicit creator approval)
            $table->boolean('use_for_ads')->default(false);
            $table->timestamp('ad_approval_at')->nullable();
            $table->foreignId('ad_approved_by')->nullable()->constrained('users')->onDelete('set null');
            
            // Audit fields
            $table->foreignId('created_by')->nullable()->constrained('users')->onDelete('set null');
            $table->foreignId('updated_by')->nullable()->constrained('users')->onDelete('set null');
            
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['creator_id', 'platform']);
            $table->index(['creator_id', 'status']);
            $table->index('status');
            $table->index('token_expires_at');
            $table->unique(['creator_id', 'platform', 'platform_user_id'], 'unique_creator_platform_account');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('creator_social_connections');
    }
};