# Brand Admin Sidebar Navigation

Complete documentation for the Brand Admin left sidebar navigation system.

## Overview

The Brand Admin sidebar is a comprehensive navigation system designed specifically for brand administrators. It provides quick access to all brand-level management features, campaign controls, and social media tools.

## Features

### ✅ Core Features
- **18 Navigation Items** in logical order
- **Role-Based Access** - Only visible to `brand_admin` role
- **Collapsible/Expandable** sidebar with persistent state
- **Badge System** for real-time notifications
- **Quick Search** (Ctrl+K) for fast navigation
- **Mobile Responsive** with hamburger menu
- **Keyboard Navigation** support
- **Accessibility Compliant** (ARIA roles, focus states)
- **Tooltip Support** in collapsed mode

### 🎨 UI/UX Features
- Smooth animations and transitions
- Active menu item highlighting
- Hover effects and visual feedback
- Gradient background design
- Icon-based navigation
- Submenu support (Settings)

## Installation

### 1. Files Created

```
resources/views/components/brand-sidebar.blade.php
resources/views/layouts/brand-admin.blade.php
app/Http/Controllers/Brand/SidebarController.php
docs/BRAND_ADMIN_SIDEBAR.md
```

### 2. Routes Added

```php
// In routes/web.php
Route::get('/api/sidebar/badges', [SidebarController::class, 'getBadges'])
    ->name('brand.sidebar.badges');
Route::get('/api/sidebar/search', [SidebarController::class, 'quickSearch'])
    ->name('brand.sidebar.search');
```

## Usage

### Basic Usage

To use the sidebar in your brand admin pages, use the `brand-admin` layout:

```blade
<x-brand-admin-layout>
    <x-slot name="header">
        Dashboard
    </x-slot>

    <x-slot name="breadcrumbs">
        <li><a href="{{ route('brand.dashboard') }}">Home</a></li>
        <li>Dashboard</li>
    </x-slot>

    <!-- Your page content here -->
    <div class="container">
        <h1>Welcome to Brand Dashboard</h1>
    </div>
</x-brand-admin-layout>
```

### Direct Component Usage

If you need more control, use the component directly:

```blade
<x-brand-sidebar />

<div class="main-content" style="margin-left: 280px;">
    <!-- Your content -->
</div>
```

## Navigation Structure

### Main Navigation Items (in order)

1. **Dashboard** - KPI cards, overview metrics
2. **Social Accounts** - Connect/manage OAuth providers
3. **Creators** - Discovery, management, invitations
4. **Campaigns** - Create, manage, track campaigns (with badge)
5. **Proposals** - Full lifecycle management (with badge)
6. **Content Ideas** - AI-generated suggestions
7. **Scheduled Posts** - Calendar view, approval workflow
8. **Reels** - AI reel creation and management
9. **Ads Manager** - A/B testing, performance tracking
10. **Analytics** - Brand-wide charts and KPIs
11. **Competitor Insights** - Track competitors, alerts
12. **Messaging** - Chat with creators (with badge)
13. **Notifications** - Email + in-app notifications (with badge)
14. **Localization** - Multilingual content translation
15. **Support Bot** - FAQ management, handoff to human
16. **Settings** (expandable submenu):
    - Brand Profile
    - Team Management
    - API Keys
17. **Help & Docs** - Tutorials and documentation
18. **Account Menu** (bottom):
    - Profile
    - Audit Logs
    - Logout

## Badge System

### Real-Time Badges

The sidebar displays real-time notification badges for:

- **Pending Proposals** (yellow badge)
- **Active Campaigns** (green badge)
- **Unread Messages** (red badge)
- **Notifications** (red badge)

### Badge API

Badges are fetched from:
```
GET /brand/api/sidebar/badges
```

Response format:
```json
{
    "success": true,
    "badges": {
        "pendingProposals": 3,
        "activeCampaigns": 5,
        "unreadMessages": 12,
        "notifications": 8
    }
}
```

Badges auto-refresh every 30 seconds.

## Quick Search

### Activation
- Press `Ctrl+K` (or `Cmd+K` on Mac)
- Click the "Quick Search" button in sidebar

### Features
- Searches navigation items
- Searches campaigns, creators, proposals via API
- Real-time results as you type
- Keyboard navigable

### Search API

```
GET /brand/api/sidebar/search?query={searchTerm}
```

Response format:
```json
{
    "success": true,
    "results": [
        {
            "name": "Campaign Name",
            "category": "Campaign",
            "url": "/brand/campaigns/123",
            "meta": "Status: active"
        }
    ]
}
```

## Customization

### Changing Sidebar Width

Edit the component:
```javascript
// In brand-sidebar.blade.php
:style="{'width': sidebarCollapsed ? '80px' : '320px'}" // Change 320px
```

### Adding New Navigation Items

1. Add the HTML link in the navigation section:
```blade
<a href="{{ route('brand.new-feature') }}" 
   @click="setActive('new-feature')"
   :class="isActive('new-feature') ? 'bg-blue-600 text-white' : 'text-gray-300 hover:bg-gray-700'"
   class="nav-item flex items-center space-x-3 px-3 py-2.5 rounded-lg transition-all duration-200 group"
   role="menuitem">
    <svg class="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <!-- Your icon SVG path -->
    </svg>
    <span x-show="!sidebarCollapsed" class="text-sm font-medium">New Feature</span>
</a>
```

2. Add to `allNavItems` array for search:
```javascript
allNavItems: [
    // ... existing items
    { name: 'New Feature', url: '{{ route("brand.new-feature") }}', category: 'Navigation' }
]
```

3. Add to `setActiveFromUrl()` method:
```javascript
else if (path.includes('/new-feature')) this.activeItem = 'new-feature';
```

### Adding Badges to New Items

```blade
<span x-show="!sidebarCollapsed && badges.newFeatureCount > 0" 
      class="ml-auto bg-blue-500 text-white text-xs px-2 py-0.5 rounded-full"
      x-text="badges.newFeatureCount"></span>
```

Update the badge API to include the new count.

## Mobile Responsiveness

### Breakpoints
- **Desktop** (≥1024px): Full sidebar visible
- **Mobile** (<1024px): Hamburger menu with overlay

### Mobile Behavior
- Sidebar hidden by default
- Hamburger button in top-left
- Click overlay to close
- Full sidebar slides in from left

## Keyboard Navigation

### Shortcuts
- `Ctrl+K` / `Cmd+K` - Open quick search
- `Escape` - Close modals/search
- `Tab` - Navigate through items
- `Enter` - Activate focused item

### Focus Management
- Visible focus indicators
- Logical tab order
- Trapped focus in modals

## Accessibility

### ARIA Attributes
- `role="navigation"` on nav element
- `role="menuitem"` on nav links
- `aria-label` for icon-only buttons
- `aria-expanded` for expandable menus
- `aria-modal` for modals

### Screen Reader Support
- Descriptive labels
- Hidden text for icons
- Semantic HTML structure

### Keyboard Support
- All interactive elements keyboard accessible
- Logical focus order
- Visual focus indicators

## State Persistence

### LocalStorage Keys
- `sidebarCollapsed` - Sidebar collapsed state
- `activeNavItem` - Currently active menu item

### Behavior
- State persists across page reloads
- Per-browser storage
- Automatic sync on state change

## Styling

### Color Scheme
- Background: Gradient from gray-900 to gray-800
- Active item: Blue-600
- Hover: Gray-700
- Text: White/Gray-300

### Customizing Colors

Edit the Tailwind classes in the component:
```blade
<!-- Change background gradient -->
class="bg-gradient-to-b from-gray-900 to-gray-800"

<!-- Change active state -->
:class="isActive('item') ? 'bg-blue-600 text-white' : 'text-gray-300'"
```

## Performance

### Optimizations
- Badge API calls cached for 30 seconds
- Debounced search input
- Lazy-loaded search results
- Minimal re-renders with Alpine.js

### Best Practices
- Keep badge counts lightweight
- Limit search results to 20 items
- Use CDN for icons if needed

## Troubleshooting

### Sidebar Not Showing
1. Check user has `brand_admin` role
2. Verify layout is using `@hasrole('brand_admin')`
3. Check Alpine.js is loaded

### Badges Not Updating
1. Verify API route is accessible
2. Check CSRF token in headers
3. Inspect browser console for errors
4. Verify database queries in controller

### Search Not Working
1. Check search API endpoint
2. Verify query parameter encoding
3. Check database indexes for performance
4. Review error logs

### Mobile Menu Issues
1. Verify viewport meta tag
2. Check z-index conflicts
3. Test touch events
4. Validate responsive breakpoints

## Security Considerations

### Role-Based Access
- Sidebar only renders for `brand_admin` role
- All routes protected by middleware
- API endpoints require authentication

### XSS Prevention
- All user input escaped
- Blade templating auto-escapes
- CSP headers recommended

### CSRF Protection
- All POST requests include CSRF token
- API calls include X-CSRF-TOKEN header

## Testing

### Manual Testing Checklist
- [ ] Sidebar visible for brand_admin
- [ ] Sidebar hidden for other roles
- [ ] All navigation links work
- [ ] Badges display correct counts
- [ ] Quick search returns results
- [ ] Collapse/expand persists
- [ ] Mobile menu functions
- [ ] Keyboard navigation works
- [ ] Screen reader accessible
- [ ] No console errors

### Automated Testing

Example test:
```php
// tests/Feature/BrandSidebarTest.php
public function test_sidebar_visible_for_brand_admin()
{
    $user = User::factory()->create();
    $user->assignRole('brand_admin');
    
    $response = $this->actingAs($user)
        ->get(route('brand.dashboard'));
    
    $response->assertSee('Brand Admin');
    $response->assertSee('Dashboard');
}
```

## Future Enhancements

### Planned Features
- [ ] Drag-and-drop menu reordering
- [ ] Custom menu item creation
- [ ] Theme customization UI
- [ ] Notification center panel
- [ ] Recent items history
- [ ] Favorites/bookmarks
- [ ] Multi-language support
- [ ] Dark/light mode toggle

## Support

For issues or questions:
1. Check this documentation
2. Review error logs
3. Inspect browser console
4. Contact development team

## Changelog

### Version 1.0.0 (2025-01-30)
- Initial release
- 18 navigation items
- Badge system
- Quick search
- Mobile responsive
- Accessibility compliant
- Keyboard navigation