# Marketing Frontend - Implementation Guide

## Quick Start Integration

Follow these steps to integrate the new marketing frontend components into your Laravel application.

---

## Step 1: Update Marketing Layout

Add the new CSS and JS files to [`resources/views/layouts/marketing.blade.php`](resources/views/layouts/marketing.blade.php:34):

```blade
<!-- Replace line 34 with: -->
@vite(['resources/css/app.css', 'resources/css/design-tokens.css', 'resources/js/app.js', 'resources/js/analytics.js'])
```

Add cookie consent before closing `</body>` tag (around line 230):

```blade
<!-- Add before </body> -->
<x-marketing.cookie-consent />
```

Add search dropdown to navigation (around line 113):

```blade
<!-- Add after navigation links, before auth section -->
<x-marketing.search-dropdown />
```

---

## Step 2: Update MarketingController

Add new methods to `app/Http/Controllers/MarketingController.php`:

```php
<?php

namespace App\Http\Controllers;

use App\Models\Plan;
use Illuminate\Http\Request;

class MarketingController extends Controller
{
    public function home(Request $request)
    {
        $selectedRole = $request->get('role', 'brand');
        return view('marketing.home', compact('selectedRole'));
    }

    public function pricing(Request $request)
    {
        $plans = Plan::where('is_active', true)->orderBy('price')->get();
        $interval = $request->get('interval', 'monthly');
        $selectedRole = $request->get('role', 'all');
        
        return view('marketing.pricing', compact('plans', 'interval', 'selectedRole'));
    }

    public function features()
    {
        return view('marketing.features');
    }

    public function demo()
    {
        return view('marketing.demo');
    }

    public function help()
    {
        return view('marketing.help');
    }

    public function onboardingBrand()
    {
        return view('marketing.onboarding.brand');
    }

    public function onboardingCreator()
    {
        return view('marketing.onboarding.creator');
    }

    public function onboardingAgency()
    {
        return view('marketing.onboarding.agency');
    }

    public function contactSales(Request $request)
    {
        // Handle contact sales form submission
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'company' => 'required|string|max:255',
            'message' => 'required|string'
        ]);

        // Send email to sales team
        // Mail::to('sales@aibusinessmarket.com')->send(new ContactSales($validated));

        return back()->with('success', 'Thank you! Our sales team will contact you shortly.');
    }
}
```

---

## Step 3: Create Missing Onboarding Pages

Create Creator onboarding page:

```bash
cp resources/views/marketing/onboarding/brand.blade.php resources/views/marketing/onboarding/creator.blade.php
```

Then update the content for creator-specific messaging.

Create Agency onboarding page:

```bash
cp resources/views/marketing/onboarding/brand.blade.php resources/views/marketing/onboarding/agency.blade.php
```

Then update the content for agency-specific messaging.

---

## Step 4: Update Vite Configuration

Ensure `vite.config.js` includes the new files:

```javascript
import { defineConfig } from 'vite';
import laravel from 'laravel-vite-plugin';

export default defineConfig({
    plugins: [
        laravel({
            input: [
                'resources/css/app.css',
                'resources/css/design-tokens.css',
                'resources/js/app.js',
                'resources/js/analytics.js'
            ],
            refresh: true,
        }),
    ],
});
```

---

## Step 5: Configure Analytics

Add to `.env`:

```env
GOOGLE_ANALYTICS_ID=G-XXXXXXXXXX
FACEBOOK_PIXEL_ID=XXXXXXXXXX
```

Add to `config/services.php`:

```php
'google_analytics' => [
    'id' => env('GOOGLE_ANALYTICS_ID'),
],

'facebook_pixel' => [
    'id' => env('FACEBOOK_PIXEL_ID'),
],
```

---

## Step 6: Create Search API Endpoint

Add search route to `routes/web.php`:

```php
Route::get('/api/search', [MarketingController::class, 'search'])->name('api.search');
```

Add search method to MarketingController:

```php
public function search(Request $request)
{
    $query = $request->get('q', '');
    
    if (strlen($query) < 2) {
        return response()->json(['results' => []]);
    }
    
    // Search across pages, docs, features
    $results = [
        // Example results
        [
            'id' => 1,
            'title' => 'Getting Started Guide',
            'description' => 'Learn how to set up your account and create your first campaign',
            'url' => route('marketing.help'),
            'type' => 'doc'
        ],
        // Add more search logic here
    ];
    
    return response()->json(['results' => $results]);
}
```

---

## Step 7: Build Assets

```bash
# Install dependencies
npm install

# Build for development
npm run dev

# Build for production
npm run build
```

---

## Step 8: Upload Marketing Assets

Upload the following images to `public/images/marketing/`:

### Required Images:
- `logo.svg` - Main logo
- `og-image.jpg` (1200x630) - OpenGraph image
- `twitter-card.jpg` (1200x675) - Twitter card
- `demo-thumbnail.jpg` (1920x1080) - Demo video thumbnail
- `dashboard-screenshot.png` - Dashboard screenshot
- `campaigns-screenshot.png` - Campaigns screenshot
- `ai-tools-screenshot.png` - AI tools screenshot
- `analytics-screenshot.png` - Analytics screenshot

### Directory Structure:
```
public/images/marketing/
├── logos/
│   ├── logo.svg
│   └── favicon.ico
├── screenshots/
│   ├── dashboard-screenshot.png
│   ├── campaigns-screenshot.png
│   ├── ai-tools-screenshot.png
│   └── analytics-screenshot.png
├── og-images/
│   ├── og-image.jpg
│   └── twitter-card.jpg
└── demo-thumbnail.jpg
```

---

## Step 9: Test Everything

### Manual Testing Checklist:

- [ ] Visit `/` - Landing page loads correctly
- [ ] Visit `/pricing` - Pricing page with toggle works
- [ ] Visit `/demo` - Demo page with video player
- [ ] Visit `/help` - Help center with search
- [ ] Visit `/onboarding/brand` - Onboarding wizard works
- [ ] Click pricing plan CTA - Checkout modal opens
- [ ] Complete checkout - Success page displays
- [ ] Visit `/subscription/manage` - Subscription page works
- [ ] Cookie consent banner appears
- [ ] Search dropdown works
- [ ] Mobile responsive on all pages
- [ ] Analytics events fire (check browser console)

### Browser Testing:
- [ ] Chrome (desktop & mobile)
- [ ] Firefox
- [ ] Safari (desktop & mobile)
- [ ] Edge

### Performance Testing:
```bash
# Run Lighthouse audit
npm run lighthouse

# Or manually in Chrome DevTools
# Open DevTools > Lighthouse > Generate Report
```

Target scores:
- Performance: ≥ 80
- Accessibility: ≥ 90
- Best Practices: ≥ 90
- SEO: ≥ 90

---

## Step 10: Deploy to Production

```bash
# Clear all caches
php artisan optimize:clear

# Build production assets
npm run build

# Optimize images
npm run optimize-images

# Cache routes and config
php artisan optimize

# Deploy
git add .
git commit -m "Add marketing frontend implementation"
git push origin main
```

---

## Troubleshooting

### Issue: Blade compilation errors

**Solution**: Clear view cache
```bash
php artisan view:clear
```

### Issue: Assets not loading

**Solution**: Rebuild assets
```bash
npm run build
php artisan optimize:clear
```

### Issue: Analytics not tracking

**Solution**: Check browser console for errors and verify:
1. Google Analytics ID is set in `.env`
2. Cookie consent is accepted
3. `analytics.js` is loaded

### Issue: Razorpay modal not opening

**Solution**: Verify:
1. Razorpay keys are set in `.env`
2. Razorpay script is loaded in checkout page
3. No JavaScript errors in console

---

## Component Usage Examples

### Using Checkout Modal

```blade
<x-marketing.checkout-modal :plan="$plan">
    Start {{ $plan->trial_days }}-Day Trial
</x-marketing.checkout-modal>
```

### Using Structured Data

```blade
<x-marketing.structured-data 
    type="Product" 
    :data="['name' => 'AI Business Market', 'price' => 999]" 
/>
```

### Tracking Analytics Events

```javascript
// In your Blade template
<button onclick="window.tracker.ctaClick('Start Trial', 'Hero Section')">
    Start Trial
</button>

// Or in JavaScript
window.tracker.pricingView('plan-pro', 'brand');
window.tracker.checkoutInitiated('plan-pro', 'Pro Plan', 999);
```

---

## Support

If you encounter any issues:

1. Check the [Marketing Frontend Documentation](MARKETING_FRONTEND.md)
2. Review Laravel logs: `storage/logs/laravel.log`
3. Check browser console for JavaScript errors
4. Contact: dev@aibusinessmarket.com

---

**Last Updated**: 2024-01-15  
**Version**: 1.0.0