/**
 * Super Admin Sidebar JavaScript
 * Handles global search, keyboard navigation, confirmation modals, and mobile responsiveness
 */

(function() {
    'use strict';

    // Global Search Functionality
    class GlobalSearch {
        constructor() {
            this.searchInput = document.getElementById('sidebar-search');
            this.searchResults = document.getElementById('search-results');
            this.debounceTimer = null;
            this.minSearchLength = 2;
            
            if (this.searchInput) {
                this.init();
            }
        }

        init() {
            this.searchInput.addEventListener('input', (e) => this.handleSearch(e));
            this.searchInput.addEventListener('keydown', (e) => this.handleKeyboard(e));
            
            // Close results when clicking outside
            document.addEventListener('click', (e) => {
                if (!e.target.closest('.sidebar-search-wrapper')) {
                    this.hideResults();
                }
            });
        }

        handleSearch(e) {
            clearTimeout(this.debounceTimer);
            const query = e.target.value.trim();

            if (query.length < this.minSearchLength) {
                this.hideResults();
                return;
            }

            this.debounceTimer = setTimeout(() => {
                this.performSearch(query);
            }, 300);
        }

        async performSearch(query) {
            try {
                const response = await fetch(`/superadmin/search?q=${encodeURIComponent(query)}`, {
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json'
                    }
                });

                if (!response.ok) throw new Error('Search failed');

                const data = await response.json();
                this.displayResults(data.results);
            } catch (error) {
                console.error('Search error:', error);
                this.displayError();
            }
        }

        displayResults(results) {
            if (!results || results.length === 0) {
                this.searchResults.innerHTML = '<div class="search-no-results">No results found</div>';
                this.showResults();
                return;
            }

            const html = results.map(result => `
                <a href="${result.url}" class="search-result-item">
                    <div class="search-result-title">
                        <i class="${result.icon || 'fas fa-circle'}"></i>
                        ${this.escapeHtml(result.title)}
                    </div>
                    <div class="search-result-meta">${this.escapeHtml(result.type)}</div>
                </a>
            `).join('');

            this.searchResults.innerHTML = html;
            this.showResults();
        }

        displayError() {
            this.searchResults.innerHTML = '<div class="search-no-results text-danger">Search error occurred</div>';
            this.showResults();
        }

        showResults() {
            this.searchResults.classList.add('show');
        }

        hideResults() {
            this.searchResults.classList.remove('show');
        }

        handleKeyboard(e) {
            if (e.key === 'Escape') {
                this.hideResults();
                this.searchInput.blur();
            }
        }

        escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    }

    // Confirmation Modal for Destructive Actions
    class ConfirmationModal {
        constructor() {
            this.modal = null;
            this.createModal();
            this.bindEvents();
        }

        createModal() {
            const modalHtml = `
                <div id="confirmation-modal" class="confirmation-modal" role="dialog" aria-labelledby="modal-title" aria-modal="true">
                    <div class="confirmation-modal-content">
                        <div class="confirmation-modal-header">
                            <i class="confirmation-modal-icon fas fa-exclamation-triangle danger"></i>
                            <h3 id="modal-title" class="confirmation-modal-title">Confirm Action</h3>
                        </div>
                        <div class="confirmation-modal-body" id="modal-body">
                            Are you sure you want to perform this action?
                        </div>
                        <div class="confirmation-modal-footer">
                            <button type="button" class="btn btn-secondary" id="modal-cancel">Cancel</button>
                            <button type="button" class="btn btn-danger" id="modal-confirm">Confirm</button>
                        </div>
                    </div>
                </div>
            `;
            
            document.body.insertAdjacentHTML('beforeend', modalHtml);
            this.modal = document.getElementById('confirmation-modal');
        }

        bindEvents() {
            // Bind destructive action links
            document.addEventListener('click', (e) => {
                const target = e.target.closest('[data-confirm]');
                if (target) {
                    e.preventDefault();
                    this.show(target);
                }
            });

            // Modal buttons
            document.getElementById('modal-cancel').addEventListener('click', () => this.hide());
            document.getElementById('modal-confirm').addEventListener('click', () => this.confirm());
            
            // Close on overlay click
            this.modal.addEventListener('click', (e) => {
                if (e.target === this.modal) {
                    this.hide();
                }
            });

            // Keyboard support
            document.addEventListener('keydown', (e) => {
                if (this.modal.classList.contains('show')) {
                    if (e.key === 'Escape') {
                        this.hide();
                    } else if (e.key === 'Enter') {
                        this.confirm();
                    }
                }
            });
        }

        show(element) {
            this.currentElement = element;
            const message = element.dataset.confirm || 'Are you sure you want to perform this action?';
            const title = element.dataset.confirmTitle || 'Confirm Action';
            const type = element.dataset.confirmType || 'danger';

            document.getElementById('modal-title').textContent = title;
            document.getElementById('modal-body').textContent = message;
            
            const icon = this.modal.querySelector('.confirmation-modal-icon');
            icon.className = `confirmation-modal-icon fas ${type === 'warning' ? 'fa-exclamation-circle warning' : 'fa-exclamation-triangle danger'}`;

            this.modal.classList.add('show');
            document.getElementById('modal-confirm').focus();
        }

        hide() {
            this.modal.classList.remove('show');
            this.currentElement = null;
        }

        confirm() {
            if (this.currentElement) {
                const href = this.currentElement.href;
                const form = this.currentElement.closest('form');
                
                if (form) {
                    form.submit();
                } else if (href) {
                    window.location.href = href;
                }
            }
            this.hide();
        }
    }

    // Keyboard Navigation
    class KeyboardNavigation {
        constructor() {
            this.init();
        }

        init() {
            document.addEventListener('keydown', (e) => {
                // Ctrl+K or Cmd+K for search
                if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
                    e.preventDefault();
                    const searchInput = document.getElementById('sidebar-search');
                    if (searchInput) {
                        searchInput.focus();
                    }
                }

                // Arrow key navigation in sidebar
                if (document.activeElement.closest('.sidebar')) {
                    this.handleArrowKeys(e);
                }
            });
        }

        handleArrowKeys(e) {
            const currentLink = document.activeElement;
            if (!currentLink.classList.contains('nav-link')) return;

            let nextLink = null;

            switch(e.key) {
                case 'ArrowDown':
                    e.preventDefault();
                    nextLink = this.getNextLink(currentLink);
                    break;
                case 'ArrowUp':
                    e.preventDefault();
                    nextLink = this.getPreviousLink(currentLink);
                    break;
                case 'ArrowRight':
                    e.preventDefault();
                    this.expandSubmenu(currentLink);
                    break;
                case 'ArrowLeft':
                    e.preventDefault();
                    this.collapseSubmenu(currentLink);
                    break;
            }

            if (nextLink) {
                nextLink.focus();
            }
        }

        getNextLink(current) {
            const allLinks = Array.from(document.querySelectorAll('.sidebar .nav-link'));
            const currentIndex = allLinks.indexOf(current);
            return allLinks[currentIndex + 1] || allLinks[0];
        }

        getPreviousLink(current) {
            const allLinks = Array.from(document.querySelectorAll('.sidebar .nav-link'));
            const currentIndex = allLinks.indexOf(current);
            return allLinks[currentIndex - 1] || allLinks[allLinks.length - 1];
        }

        expandSubmenu(link) {
            const parent = link.closest('.nav-item');
            if (parent && parent.querySelector('.nav-treeview')) {
                parent.classList.add('menu-open');
            }
        }

        collapseSubmenu(link) {
            const parent = link.closest('.nav-item');
            if (parent && parent.classList.contains('menu-open')) {
                parent.classList.remove('menu-open');
            }
        }
    }

    // Mobile Sidebar Toggle
    class MobileSidebar {
        constructor() {
            this.sidebar = document.querySelector('.main-sidebar');
            this.overlay = this.createOverlay();
            this.init();
        }

        createOverlay() {
            const overlay = document.createElement('div');
            overlay.className = 'sidebar-overlay';
            document.body.appendChild(overlay);
            return overlay;
        }

        init() {
            // Toggle button
            const toggleBtn = document.querySelector('[data-widget="pushmenu"]');
            if (toggleBtn) {
                toggleBtn.addEventListener('click', () => this.toggle());
            }

            // Close on overlay click
            this.overlay.addEventListener('click', () => this.hide());

            // Close on escape key
            document.addEventListener('keydown', (e) => {
                if (e.key === 'Escape' && this.isVisible()) {
                    this.hide();
                }
            });
        }

        toggle() {
            if (this.isVisible()) {
                this.hide();
            } else {
                this.show();
            }
        }

        show() {
            if (window.innerWidth <= 768) {
                this.sidebar.classList.add('show');
                this.overlay.classList.add('show');
                document.body.style.overflow = 'hidden';
            }
        }

        hide() {
            this.sidebar.classList.remove('show');
            this.overlay.classList.remove('show');
            document.body.style.overflow = '';
        }

        isVisible() {
            return this.sidebar.classList.contains('show');
        }
    }

    // Sidebar State Persistence
    class SidebarState {
        constructor() {
            this.storageKey = 'superadmin_sidebar_state';
            this.init();
        }

        init() {
            this.restoreState();
            this.bindEvents();
        }

        restoreState() {
            const state = this.getState();
            if (state.collapsed) {
                document.body.classList.add('sidebar-collapse');
            }

            // Restore open submenus
            if (state.openMenus) {
                state.openMenus.forEach(menuId => {
                    const menu = document.querySelector(`[data-menu-id="${menuId}"]`);
                    if (menu) {
                        menu.classList.add('menu-open');
                    }
                });
            }
        }

        bindEvents() {
            // Save collapse state
            const toggleBtn = document.querySelector('[data-widget="pushmenu"]');
            if (toggleBtn) {
                toggleBtn.addEventListener('click', () => {
                    setTimeout(() => {
                        this.saveState();
                    }, 100);
                });
            }

            // Save submenu state
            document.querySelectorAll('.sidebar .nav-item.has-treeview').forEach(item => {
                item.addEventListener('click', () => {
                    setTimeout(() => {
                        this.saveState();
                    }, 100);
                });
            });
        }

        getState() {
            const stored = localStorage.getItem(this.storageKey);
            return stored ? JSON.parse(stored) : { collapsed: false, openMenus: [] };
        }

        saveState() {
            const state = {
                collapsed: document.body.classList.contains('sidebar-collapse'),
                openMenus: Array.from(document.querySelectorAll('.sidebar .nav-item.menu-open'))
                    .map(item => item.dataset.menuId)
                    .filter(Boolean)
            };
            localStorage.setItem(this.storageKey, JSON.stringify(state));
        }
    }

    // Initialize all components when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    function init() {
        // Only initialize for super admin users
        if (document.body.classList.contains('superadmin') || document.querySelector('[data-superadmin]')) {
            new GlobalSearch();
            new ConfirmationModal();
            new KeyboardNavigation();
            new MobileSidebar();
            new SidebarState();

            console.log('Super Admin Sidebar initialized');
        }
    }
})();