@extends('adminlte::page')

@section('title', 'Content Ideas')

@section('content')
<div class="content-header">
    <div class="container-fluid">
        <div class="row mb-2">
            <div class="col-sm-6">
                <h1 class="m-0">Content Ideas</h1>
            </div>
            <div class="col-sm-6">
                <ol class="breadcrumb float-sm-right">
                    <li class="breadcrumb-item"><a href="{{ route('brand.dashboard') }}">Home</a></li>
                    <li class="breadcrumb-item active">Content Ideas</li>
                </ol>
            </div>
        </div>
    </div>
</div>

<div class="content">
    <div class="container-fluid">
        @if(session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            {{ session('success') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
        @endif

        @if(session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            {{ session('error') }}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
        @endif

        <div class="row mb-3">
            <div class="col-12">
                <a href="{{ route('brand.content-ideas.create') }}" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Generate New Content Ideas
                </a>
            </div>
        </div>

        @if($contentIdeas->count() > 0)
        <div class="row">
            @foreach($contentIdeas as $idea)
            <div class="col-md-6 col-lg-4 mb-4">
                <div class="card h-100">
                    <div class="card-body">
                        <h5 class="card-title">Content Idea</h5>
                        <p class="card-text">{{ $idea->caption }}</p>
                        
                        @if($idea->hashtags)
                        <div class="mb-3">
                            <strong>Hashtags:</strong>
                            <div>
                                @foreach($idea->hashtags as $hashtag)
                                <span class="badge badge-info">{{ $hashtag }}</span>
                                @endforeach
                            </div>
                        </div>
                        @endif
                        
                        @if($idea->creative_hints)
                        <div class="mb-3">
                            <strong>Creative Hints:</strong>
                            <p class="text-muted">{{ $idea->creative_hints }}</p>
                        </div>
                        @endif
                        
                        @if($idea->campaign)
                        <div class="mb-3">
                            <strong>Campaign:</strong>
                            <span class="badge badge-primary">{{ $idea->campaign->name }}</span>
                        </div>
                        @endif
                        
                        <div class="mb-3">
                            <strong>Status:</strong>
                            <span class="badge 
                                @if($idea->status === 'draft') badge-secondary
                                @elseif($idea->status === 'scheduled') badge-warning
                                @else badge-light
                                @endif">
                                {{ ucfirst($idea->status) }}
                            </span>
                        </div>
                        
                        <div class="mb-3">
                            <small class="text-muted">
                                Created by: {{ $idea->user->name ?? 'Unknown' }}
                                on {{ $idea->created_at->format('M j, Y H:i') }}
                            </small>
                        </div>
                    </div>
                    <div class="card-footer">
                        <div class="btn-group" role="group">
                            <button type="button" class="btn btn-sm btn-outline-primary copy-idea" 
                                    data-id="{{ $idea->id }}" data-toggle="tooltip" title="Copy">
                                <i class="fas fa-copy"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-secondary save-to-draft" 
                                    data-id="{{ $idea->id }}" data-toggle="tooltip" title="Save to Draft">
                                <i class="fas fa-save"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-success schedule-post" 
                                    data-id="{{ $idea->id }}" data-toggle="tooltip" title="Schedule Post">
                                <i class="fas fa-calendar-plus"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            @endforeach
        </div>
        
        <div class="row">
            <div class="col-12">
                {{ $contentIdeas->links() }}
            </div>
        </div>
        @else
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body text-center">
                        <p class="card-text">No content ideas found.</p>
                        <a href="{{ route('brand.content-ideas.create') }}" class="btn btn-primary">
                            Generate Your First Content Idea
                        </a>
                    </div>
                </div>
            </div>
        </div>
        @endif
    </div>
</div>

<!-- Schedule Post Modal -->
<div class="modal fade" id="scheduleModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <form id="scheduleForm" method="POST">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title">Schedule Content Post</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-group">
                        <label for="scheduled_at">Scheduled Date & Time</label>
                        <input type="datetime-local" class="form-control" id="scheduled_at" name="scheduled_at" required>
                    </div>
                    <div class="form-group">
                        <label for="timezone">Timezone</label>
                        <select class="form-control" id="timezone" name="timezone" required>
                            <option value="UTC">UTC</option>
                            <option value="Asia/Calcutta">Asia/Calcutta</option>
                            <option value="America/New_York">America/New_York</option>
                            <option value="Europe/London">Europe/London</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">Schedule</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Copy idea functionality
        document.querySelectorAll('.copy-idea').forEach(button => {
            button.addEventListener('click', function() {
                const ideaId = this.getAttribute('data-id');
                
                fetch(`/brand/content-ideas/${ideaId}/copy`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                })
                .then(response => response.json())
                .then(data => {
                    // Copy to clipboard
                    const textToCopy = `${data.caption}\n\n${data.hashtags.join(' ')}\n\n${data.creative_hints}`;
                    navigator.clipboard.writeText(textToCopy)
                        .then(() => {
                            showSyncToast('Content idea copied to clipboard!', 'success');
                        })
                        .catch(err => {
                            showSyncToast('Failed to copy content idea.', 'danger');
                        });
                })
                .catch(error => {
                    showSyncToast('Failed to copy content idea.', 'danger');
                });
            });
        });
        
        // Save to draft functionality
        document.querySelectorAll('.save-to-draft').forEach(button => {
            button.addEventListener('click', function() {
                const ideaId = this.getAttribute('data-id');
                
                fetch(`/brand/content-ideas/${ideaId}/save-to-draft`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                })
                .then(response => response.json())
                .then(data => {
                    showSyncToast('Content idea saved to draft!', 'success');
                    // Reload the page to update the status
                    setTimeout(() => {
                        window.location.reload();
                    }, 1000);
                })
                .catch(error => {
                    showSyncToast('Failed to save content idea to draft.', 'danger');
                });
            });
        });
        
        // Schedule post functionality
        document.querySelectorAll('.schedule-post').forEach(button => {
            button.addEventListener('click', function() {
                const ideaId = this.getAttribute('data-id');
                const form = document.getElementById('scheduleForm');
                
                // Update form action
                form.setAttribute('action', `/brand/content-ideas/${ideaId}/schedule-post`);
                
                // Show modal
                $('#scheduleModal').modal('show');
            });
        });
        
        // Helper function to show toast notifications
        function showSyncToast(message, type = 'info') {
            // Create toast element
            const toast = document.createElement('div');
            toast.className = `toast align-items-center text-white bg-${type} border-0`;
            toast.setAttribute('role', 'alert');
            toast.setAttribute('aria-live', 'assertive');
            toast.setAttribute('aria-atomic', 'true');
            
            toast.innerHTML = `
                <div class="d-flex">
                    <div class="toast-body">
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                </div>
            `;
            
            // Add to toast container
            const toastContainer = document.getElementById('toastContainer') || document.body;
            toastContainer.appendChild(toast);
            
            // Initialize and show toast
            const bsToast = new bootstrap.Toast(toast, {
                delay: 3000
            });
            bsToast.show();
            
            // Remove toast after it's hidden
            toast.addEventListener('hidden.bs.toast', function() {
                toast.remove();
            });
        }
    });
</script>
@endsection