@extends('adminlte::page')

@section('title', 'Brand Reels')

@section('content_header')
    <h1>Reels</h1>
@stop

@section('content')
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Scheduled Reels Calendar</h3>
                <div class="card-tools">
                    <a href="{{ route('brand.reels.create') }}" class="btn btn-primary">
                        <i class="fas fa-plus"></i> Create New Reel
                    </a>
                </div>
            </div>
            <div class="card-body">
                <div id="reels-calendar"></div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Scheduled Reels</h3>
                <div class="card-tools">
                    <form method="GET" action="{{ route('brand.reels.index') }}" class="form-inline">
                        <div class="form-group mr-2">
                            <label for="status-filter" class="mr-2">Status:</label>
                            <select name="status" id="status-filter" class="form-control">
                                <option value="">All</option>
                                <option value="draft" {{ request('status') === 'draft' ? 'selected' : '' }}>Draft</option>
                                <option value="scheduled" {{ request('status') === 'scheduled' ? 'selected' : '' }}>Scheduled</option>
                                <option value="queued" {{ request('status') === 'queued' ? 'selected' : '' }}>Queued</option>
                                <option value="posting" {{ request('status') === 'posting' ? 'selected' : '' }}>Posting</option>
                                <option value="posted" {{ request('status') === 'posted' ? 'selected' : '' }}>Posted</option>
                                <option value="failed" {{ request('status') === 'failed' ? 'selected' : '' }}>Failed</option>
                                <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                            </select>
                        </div>
                        <button type="submit" class="btn btn-default">Filter</button>
                    </form>
                </div>
            </div>
            <div class="card-body">
                @if($scheduledReels->count() > 0)
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Template</th>
                                    <th>Status</th>
                                    <th>Scheduled At</th>
                                    <th>Timezone</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($scheduledReels as $reel)
                                    <tr>
                                        <td>{{ $reel->id }}</td>
                                        <td>{{ $reel->template }}</td>
                                        <td>
                                            @if($reel->status === 'draft')
                                                <span class="badge badge-secondary">Draft</span>
                                            @elseif($reel->status === 'scheduled')
                                                <span class="badge badge-info">Scheduled</span>
                                            @elseif($reel->status === 'queued')
                                                <span class="badge badge-warning">Queued</span>
                                            @elseif($reel->status === 'posting')
                                                <span class="badge badge-primary">Posting</span>
                                            @elseif($reel->status === 'posted')
                                                <span class="badge badge-success">Posted</span>
                                            @elseif($reel->status === 'failed')
                                                <span class="badge badge-danger">Failed</span>
                                            @elseif($reel->status === 'cancelled')
                                                <span class="badge badge-dark">Cancelled</span>
                                            @endif
                                        </td>
                                        <td>{{ $reel->scheduled_at ? $reel->scheduled_at->format('Y-m-d H:i') : 'N/A' }}</td>
                                        <td>{{ $reel->timezone ?? 'N/A' }}</td>
                                        <td>{{ $reel->created_at->diffForHumans() }}</td>
                                        <td>
                                            <a href="{{ route('brand.reels.show-scheduled', $reel) }}" class="btn btn-sm btn-info">
                                                <i class="fas fa-eye"></i> View
                                            </a>
                                            @if($reel->status === 'scheduled' || $reel->status === 'draft')
                                                <form action="{{ route('brand.reels.post-now', $reel) }}" method="POST" class="d-inline">
                                                    @csrf
                                                    <button type="submit" class="btn btn-sm btn-success">
                                                        <i class="fas fa-bolt"></i> Post Now
                                                    </button>
                                                </form>
                                            @endif
                                            @if($reel->status === 'scheduled' || $reel->status === 'draft')
                                                <a href="{{ route('brand.reels.create') }}" class="btn btn-sm btn-warning">
                                                    <i class="fas fa-edit"></i> Reschedule
                                                </a>
                                            @endif
                                            @if($reel->status === 'scheduled' || $reel->status === 'draft')
                                                <form action="{{ route('brand.reels.cancel', $reel) }}" method="POST" class="d-inline">
                                                    @csrf
                                                    <button type="submit" class="btn btn-sm btn-danger">
                                                        <i class="fas fa-times"></i> Cancel
                                                    </button>
                                                </form>
                                            @endif
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                    <div class="d-flex justify-content-center">
                        {{ $scheduledReels->links() }}
                    </div>
                @else
                    <p>No scheduled reels found.</p>
                @endif
            </div>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Reel Requests</h3>
            </div>
            <div class="card-body">
                @if($reels->count() > 0)
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Creator</th>
                                    <th>Template</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Duration</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($reels as $reel)
                                    <tr>
                                        <td>{{ $reel->id }}</td>
                                        <td>{{ $reel->user->name ?? 'N/A' }}</td>
                                        <td>{{ config('reels.templates.' . $reel->template . '.name', $reel->template) }}</td>
                                        <td>
                                            <span class="badge 
                                                @if($reel->status == 'ready') badge-success
                                                @elseif($reel->status == 'processing') badge-warning
                                                @elseif($reel->status == 'failed') badge-danger
                                                @else badge-secondary
                                                @endif">
                                                {{ ucfirst($reel->status) }}
                                            </span>
                                        </td>
                                        <td>{{ $reel->created_at->format('M d, Y H:i') }}</td>
                                        <td>{{ $reel->duration ? $reel->duration . ' seconds' : 'N/A' }}</td>
                                        <td>
                                            <a href="{{ route('brand.reels.show', $reel) }}" class="btn btn-sm btn-info">
                                                <i class="fas fa-eye"></i> View
                                            </a>
                                            
                                            @if($reel->isReady())
                                                <a href="{{ route('brand.reels.download', $reel) }}" class="btn btn-sm btn-success">
                                                    <i class="fas fa-download"></i> Download
                                                </a>
                                            @elseif($reel->hasFailed())
                                                <form action="{{ route('brand.reels.retry', $reel) }}" method="POST" class="d-inline">
                                                    @csrf
                                                    <button type="submit" class="btn btn-sm btn-warning" onclick="return confirm('Retry processing this reel?')">
                                                        <i class="fas fa-redo"></i> Retry
                                                    </button>
                                                </form>
                                            @endif
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="d-flex justify-content-center">
                        {{ $reels->links() }}
                    </div>
                @else
                    <div class="text-center">
                        <p>No reels have been created yet.</p>
                        <a href="{{ route('brand.reels.create') }}" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Create Your First Reel
                        </a>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>
@stop

@section('css')
    <link rel="stylesheet" href="/css/admin_custom.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/fullcalendar@5.10.1/main.min.css">
@stop

@section('js')
    <script src="https://cdn.jsdelivr.net/npm/fullcalendar@5.10.1/main.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            var calendarEl = document.getElementById('reels-calendar');
            if (calendarEl) {
                var calendar = new FullCalendar.Calendar(calendarEl, {
                    initialView: 'dayGridMonth',
                    headerToolbar: {
                        left: 'prev,next today',
                        center: 'title',
                        right: 'dayGridMonth,timeGridWeek,timeGridDay'
                    },
                    events: [
                        @foreach($scheduledReels as $reel)
                            @if($reel->scheduled_at)
                                {
                                    title: '{{ $reel->template }}',
                                    start: '{{ $reel->scheduled_at->format('Y-m-d H:i:s') }}',
                                    classNames: ['status-{{ $reel->status }}']
                                },
                            @endif
                        @endforeach
                    ]
                });
                calendar.render();
            }
        });
    </script>
@stop