@extends('adminlte::page')

@section('title', 'Scheduled Posts')

@section('content_header')
    <h1>Scheduled Posts</h1>
@stop

@section('content')
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Scheduled Posts Calendar</h3>
                <div class="card-tools">
                    <a href="{{ route('creator.scheduled-posts.create') }}" class="btn btn-primary">
                        <i class="fas fa-plus"></i> Schedule New Post
                    </a>
                </div>
            </div>
            <div class="card-body">
                <div id="scheduled-posts-calendar"></div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Scheduled Posts</h3>
                <div class="card-tools">
                    <form method="GET" action="{{ route('creator.scheduled-posts.index') }}" class="form-inline">
                        <div class="form-group mr-2">
                            <label for="status-filter" class="mr-2">Status:</label>
                            <select name="status" id="status-filter" class="form-control">
                                <option value="">All</option>
                                <option value="draft" {{ request('status') === 'draft' ? 'selected' : '' }}>Draft</option>
                                <option value="scheduled" {{ request('status') === 'scheduled' ? 'selected' : '' }}>Scheduled</option>
                                <option value="publishing" {{ request('status') === 'publishing' ? 'selected' : '' }}>Publishing</option>
                                <option value="posted" {{ request('status') === 'posted' ? 'selected' : '' }}>Posted</option>
                                <option value="failed" {{ request('status') === 'failed' ? 'selected' : '' }}>Failed</option>
                                <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                            </select>
                        </div>
                        <div class="form-group mr-2">
                            <label for="campaign-filter" class="mr-2">Campaign:</label>
                            <select name="campaign_id" id="campaign-filter" class="form-control">
                                <option value="">All</option>
                                @foreach($campaigns as $campaign)
                                    <option value="{{ $campaign->id }}" {{ request('campaign_id') == $campaign->id ? 'selected' : '' }}>
                                        {{ $campaign->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                        <button type="submit" class="btn btn-default">Filter</button>
                    </form>
                    <button id="bulk-publish-btn" class="btn btn-success ml-2" style="display: none;">
                        <i class="fas fa-bolt"></i> Publish Selected
                    </button>
                    <button id="select-all-btn" class="btn btn-secondary ml-2">
                        <i class="fas fa-check-square"></i> Select All
                    </button>
                </div>
            </div>
            <div class="card-body">
                @if($scheduledPosts->count() > 0)
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Social Account</th>
                                    <th>Caption</th>
                                    <th>Status</th>
                                    <th>Publish Time</th>
                                    <th>Timezone</th>
                                    <th>Campaign</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <form id="bulk-publish-form" action="{{ route('creator.scheduled-posts.bulk-publish') }}" method="POST">
                                    @csrf
                                    @foreach($scheduledPosts as $post)
                                        <tr>
                                            <td>
                                                @if(in_array($post->status, ['scheduled', 'draft']))
                                                    <input type="checkbox" name="post_ids[]" value="{{ $post->id }}" class="bulk-select">
                                                @endif
                                                {{ $post->id }}
                                            </td>
                                            <td>{{ $post->socialAccount->provider ?? 'N/A' }}</td>
                                            <td>{{ Str::limit($post->caption, 50) }}</td>
                                            <td>
                                                @if($post->status === 'draft')
                                                    <span class="badge badge-secondary">Draft</span>
                                                @elseif($post->status === 'scheduled')
                                                    <span class="badge badge-info">Scheduled</span>
                                                @elseif($post->status === 'publishing')
                                                    <span class="badge badge-primary">Publishing</span>
                                                @elseif($post->status === 'posted')
                                                    <span class="badge badge-success">Posted</span>
                                                @elseif($post->status === 'failed')
                                                    <span class="badge badge-danger">Failed</span>
                                                @elseif($post->status === 'cancelled')
                                                    <span class="badge badge-dark">Cancelled</span>
                                                @endif
                                            </td>
                                            <td>{{ $post->publish_time ? $post->publish_time->format('Y-m-d H:i') : 'N/A' }}</td>
                                            <td>{{ $post->timezone ?? 'N/A' }}</td>
                                            <td>{{ $post->campaign->name ?? 'N/A' }}</td>
                                            <td>{{ $post->created_at->diffForHumans() }}</td>
                                            <td>
                                                <a href="{{ route('creator.scheduled-posts.show', $post) }}" class="btn btn-sm btn-info">
                                                    <i class="fas fa-eye"></i> View
                                                </a>
                                                @if(in_array($post->status, ['scheduled', 'draft']))
                                                    <form action="{{ route('creator.scheduled-posts.publish-now', $post) }}" method="POST" class="d-inline">
                                                        @csrf
                                                        <button type="submit" class="btn btn-sm btn-success">
                                                            <i class="fas fa-bolt"></i> Publish Now
                                                        </button>
                                                    </form>
                                                @endif
                                                @if(in_array($post->status, ['scheduled', 'draft']))
                                                    <a href="{{ route('creator.scheduled-posts.edit', $post) }}" class="btn btn-sm btn-warning">
                                                        <i class="fas fa-edit"></i> Edit
                                                    </a>
                                                @endif
                                                @if(in_array($post->status, ['scheduled', 'draft']))
                                                    <form action="{{ route('creator.scheduled-posts.cancel', $post) }}" method="POST" class="d-inline">
                                                        @csrf
                                                        <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to cancel this scheduled post?')">
                                                            <i class="fas fa-times"></i> Cancel
                                                        </button>
                                                    </form>
                                                @endif
                                                @if($post->status === 'failed')
                                                    <form action="{{ route('creator.scheduled-posts.retry', $post) }}" method="POST" class="d-inline">
                                                        @csrf
                                                        <button type="submit" class="btn btn-sm btn-warning">
                                                            <i class="fas fa-redo"></i> Retry
                                                        </button>
                                                    </form>
                                                @endif
                                                @if(in_array($post->status, ['scheduled', 'draft', 'posted']))
                                                    <form action="{{ route('creator.scheduled-posts.duplicate', $post) }}" method="POST" class="d-inline">
                                                        @csrf
                                                        <button type="submit" class="btn btn-sm btn-secondary">
                                                            <i class="fas fa-copy"></i> Duplicate
                                                        </button>
                                                    </form>
                                                @endif
                                            </td>
                                        </tr>
                                    @endforeach
                                </form>
                            </tbody>
                        </table>
                    </div>
                    <div class="d-flex justify-content-center">
                        {{ $scheduledPosts->links() }}
                    </div>
                @else
                    <p>No scheduled posts found.</p>
                @endif
            </div>
        </div>
    </div>
</div>
@stop

@section('css')
    <link rel="stylesheet" href="/css/admin_custom.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/fullcalendar@5.10.1/main.min.css">
@stop

@section('js')
    <script src="https://cdn.jsdelivr.net/npm/fullcalendar@5.10.1/main.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            var calendarEl = document.getElementById('scheduled-posts-calendar');
            if (calendarEl) {
                var calendar = new FullCalendar.Calendar(calendarEl, {
                    initialView: 'dayGridMonth',
                    headerToolbar: {
                        left: 'prev,next today',
                        center: 'title',
                        right: 'dayGridMonth,timeGridWeek,timeGridDay'
                    },
                    events: [
                        @foreach($scheduledPosts as $post)
                            @if($post->publish_time)
                                {
                                    title: '{{ Str::limit($post->caption, 30) }}',
                                    start: '{{ $post->publish_time->format('Y-m-d H:i:s') }}',
                                    classNames: ['status-{{ $post->status }}']
                                },
                            @endif
                        @endforeach
                    ]
                });
                calendar.render();
            }
            
            // Bulk publish functionality
            const bulkPublishBtn = document.getElementById('bulk-publish-btn');
            const selectAllBtn = document.getElementById('select-all-btn');
            const bulkSelects = document.querySelectorAll('.bulk-select');
            const bulkPublishForm = document.getElementById('bulk-publish-form');
            
            // Show/hide bulk publish button based on selection
            bulkSelects.forEach(checkbox => {
                checkbox.addEventListener('change', function() {
                    const checked = document.querySelectorAll('.bulk-select:checked').length;
                    bulkPublishBtn.style.display = checked > 0 ? 'inline-block' : 'none';
                });
            });
            
            // Select all functionality
            selectAllBtn.addEventListener('click', function() {
                const allChecked = document.querySelectorAll('.bulk-select:checked').length === bulkSelects.length;
                bulkSelects.forEach(checkbox => {
                    checkbox.checked = !allChecked;
                });
                bulkPublishBtn.style.display = !allChecked ? 'inline-block' : 'none';
            });
            
            // Bulk publish form submission
            bulkPublishBtn.addEventListener('click', function() {
                if (confirm('Are you sure you want to publish the selected posts?')) {
                    bulkPublishForm.submit();
                }
            });
        });
    </script>
@stop