<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Campaign;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class BrandCampaignsTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'brand_admin', 'guard_name' => 'web']);
        }
        
        if (!Role::where('name', 'creator')->where('guard_name', 'web')->exists()) {
            Role::create(['name' => 'creator', 'guard_name' => 'web']);
        }
    }

    public function test_brand_admin_can_access_campaigns_index()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.campaigns.index'));

        $response->assertStatus(200);
        $response->assertSee('Campaigns');
    }

    public function test_brand_admin_can_create_campaign()
    {
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $brandAdmin->assignRole('brand_admin');

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.campaigns.create'));

        $response->assertStatus(200);
        $response->assertSee('Create Campaign');
    }

    public function test_brand_admin_can_view_campaign_details()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::factory()->create();
        
        // Create a brand
        $brand = \App\Models\Brand::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        
        $brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
            'tenant_id' => $tenant->id,
            'brand_id' => $brand->id,
        ]);
        $brandAdmin->assignRole('brand_admin');

        $campaign = Campaign::factory()->create([
            'agency_id' => $brandAdmin->id,
            'brand_id' => $brand->id,
            'name' => 'Test Campaign',
            'status' => 'active',
        ]);

        $response = $this->actingAs($brandAdmin)
            ->get(route('brand.campaigns.show', $campaign));

        $response->assertStatus(200);
        $response->assertSee('Campaign Details');
        $response->assertSee('Test Campaign');
    }

    public function test_creator_cannot_access_brand_campaigns_pages()
    {
        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        $response = $this->actingAs($creator)
            ->get(route('brand.campaigns.index'));

        $response->assertStatus(403);

        $response = $this->actingAs($creator)
            ->get(route('brand.campaigns.create'));

        $response->assertStatus(403);
    }

    public function test_unauthenticated_user_cannot_access_campaigns()
    {
        $response = $this->get(route('brand.campaigns.index'));
        $response->assertRedirect(route('login'));

        $response = $this->get(route('brand.campaigns.create'));
        $response->assertRedirect(route('login'));
    }
}