<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\SocialAccount;
use App\Models\SocialAggregate;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class BrandCreatorProfileTest extends TestCase
{
    use RefreshDatabase;

    protected $brandAdmin;
    protected $creator;

    protected function setUp(): void
    {
        parent::setUp();

        // Create a brand
        $brand = \App\Models\Brand::factory()->create([
            'name' => 'Test Brand',
        ]);

        // Create a brand admin user
        $this->brandAdmin = User::factory()->create([
            'name' => 'Brand Admin',
            'email' => 'brand@example.com',
            'brand_id' => $brand->id,
        ]);
        $this->brandAdmin->assignRole('brand_admin');

        // Create a creator user
        $this->creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@example.com',
            'influencer_type' => 'Lifestyle',
        ]);
        $this->creator->assignRole('creator');

        // Create social accounts for the creator
        SocialAccount::factory()->create([
            'user_id' => $this->creator->id,
            'provider' => 'youtube',
            'meta' => [
                'username' => 'test_youtube',
                'followers' => 10000,
            ],
        ]);

        SocialAccount::factory()->create([
            'user_id' => $this->creator->id,
            'provider' => 'instagram',
            'meta' => [
                'username' => 'test_instagram',
                'followers' => 5000,
            ],
        ]);

        // Create social aggregates for the creator
        SocialAggregate::factory()->create([
            'user_id' => $this->creator->id,
            'key' => 'total_followers',
            'value' => [
                'value' => 15000,
                'previous' => 14000,
            ],
        ]);

        SocialAggregate::factory()->create([
            'user_id' => $this->creator->id,
            'key' => 'total_posts',
            'value' => [
                'value' => 50,
                'previous' => 45,
            ],
        ]);
    }

    /** @test */
    public function brand_admin_can_view_creator_profile()
    {
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.creators.show', $this->creator));

        $response->assertStatus(200);
        $response->assertSee($this->creator->name);
        $response->assertSee('youtube');
        $response->assertSee('instagram');
        $response->assertSee('15,000'); // Total followers
        $response->assertSee('50'); // Total posts
        $response->assertSee('Send Proposal');
    }

    /** @test */
    public function non_brand_user_cannot_access_creator_profile()
    {
        $regularUser = User::factory()->create();
        $regularUser->assignRole('user');

        $response = $this->actingAs($regularUser)
            ->get(route('brand.creators.show', $this->creator));

        $response->assertStatus(403);
    }
}