<?php

namespace Tests\Feature;

use App\Models\SocialAccount;
use App\Models\SocialAggregate;
use App\Models\SocialSyncError;
use App\Models\User;
use Database\Seeders\RolePermissionSeeder;
use Database\Seeders\SocialDemoSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Http;
use Tests\TestCase;

class CreatorSocialConditionalTest extends TestCase
{
    use RefreshDatabase, WithFaker;
    
    protected function setUp(): void
    {
        parent::setUp();
        
        // Ensure APP_KEY is set
        if (empty(config('app.key'))) {
            Artisan::call('key:generate');
        }
        
        // Use sync queue in tests
        config(['queue.default' => 'sync']);
        
        // Run role permission seeder
        $this->seed(RolePermissionSeeder::class);
        
        // Disable middleware for testing
        $this->withoutMiddleware();
        
        // Mock external API calls
        Http::fake([
            'www.googleapis.com/*' => Http::response([
                'items' => [
                    [
                        'id' => 'UC_demo_youtube',
                        'snippet' => [
                            'title' => 'Demo YouTube Channel',
                            'description' => 'This is a demo YouTube channel',
                        ],
                        'statistics' => [
                            'subscriberCount' => '15000',
                            'viewCount' => '500000',
                            'videoCount' => '120',
                        ],
                    ],
                ],
            ], 200),
            'graph.facebook.com/*' => Http::response([
                'id' => 'demo_instagram',
                'username' => 'demo_instagram',
                'name' => 'Demo Instagram Account',
                'followers_count' => 8500,
            ], 200),
            '*' => Http::response([], 200),
        ]);
    }

    /**
     * Test missing access token handled gracefully.
     *
     * @return void
     */
    public function test_missing_access_token_handled_gracefully()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Create a creator user
        $user = User::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        $user->assignRole('creator');
        
        // Acting as creator
        $this->actingAs($user);
        
        // Create a social account with no token
        $account = SocialAccount::factory()->create([
            'user_id' => $user->id,
            'provider' => 'youtube',
            'access_token_encrypted' => null,
            'is_demo' => false,
        ]);
        
        // Run the sync job
        $job = new \App\Jobs\SyncUserSocialAccounts($account);
        $job->handle();
        
        // Assert entry in social_sync_errors
        $this->assertDatabaseHas('social_sync_errors', [
            'social_account_id' => $account->id,
        ]);
    }

    /**
     * Test expired token refresh failure logs error.
     *
     * @return void
     */
    public function test_expired_token_refresh_failure_logs_error()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Create a creator user
        $user = User::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        $user->assignRole('creator');
        
        // Acting as creator
        $this->actingAs($user);
        
        // Create account with expired token and refresh token
        $account = SocialAccount::factory()->expired()->create([
            'user_id' => $user->id,
            'provider' => 'youtube',
            'refresh_token_encrypted' => 'invalid_refresh_token',
        ]);
        
        // Mock provider refresh failure
        Http::fake([
            'oauth2.googleapis.com/token' => Http::response([
                'error' => 'invalid_grant',
                'error_description' => 'Bad Request'
            ], 400),
        ]);
        
        // Run the sync job
        $job = new \App\Jobs\SyncUserSocialAccounts($account);
        $job->handle();
        
        // Assert social_sync_errors
        $this->assertDatabaseHas('social_sync_errors', [
            'social_account_id' => $account->id,
        ]);
    }

    /**
     * Test invalid provider redirects with error.
     *
     * @return void
     */
    public function test_invalid_provider_redirects_with_error()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Create a creator user
        $user = User::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        $user->assignRole('creator');
        
        // Acting as creator
        $this->actingAs($user);
        
        // Try to connect with invalid provider
        $response = $this->get(route('creator.connections.connect', ['provider' => 'invalid_provider']));
        
        // Should get error message
        $response->assertSessionHas('error');
        $response->assertRedirect(route('creator.connections.index'));
    }

    /**
     * Test successful manual connection creates account.
     *
     * @return void
     */
    public function test_successful_manual_connection_creates_account()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Create a creator user
        $user = User::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        $user->assignRole('creator');
        
        // Acting as creator
        $this->actingAs($user);
        
        // Add manual token
        $response = $this->post(route('creator.connections.manual'), [
            'provider' => 'youtube',
            'provider_user_id' => 'test_youtube_channel',
            'access_token' => 'test_access_token',
        ]);
        
        // Should be redirected with success message
        $response->assertSessionHas('success');
        $response->assertRedirect(route('creator.connections.index'));
        
        // Check that account was created
        $this->assertDatabaseHas('social_accounts', [
            'user_id' => $user->id,
            'provider' => 'youtube',
            'provider_user_id' => 'test_youtube_channel',
        ]);
    }

    /**
     * Test dashboard shows demo aggregates.
     *
     * @return void
     */
    public function test_dashboard_shows_demo_aggregates()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Seed demo data
        $this->seed(SocialDemoSeeder::class);
        
        // Get the demo creator user
        $user = User::where('email', 'creator1@demo.local')->first();
        $this->assertNotNull($user);
        
        // Acting as creator
        $this->actingAs($user);
        
        // Visit dashboard
        $response = $this->get(route('creator.dashboard'));
        
        // Should see dashboard with social account information
        $response->assertStatus(200);
        $response->assertSee('Connection Status');
        
        // Check for total followers (should be from demo data)
        // From SocialDemoSeeder: 15000 (YouTube) + 8500 (Instagram) + 3200 (Twitter) + 2400 (LinkedIn) + 12000 (TikTok) = 41100
        $response->assertSee('41,100'); // Formatted with commas
    }

    /**
     * Test creator can view provider-specific dashboard.
     *
     * @return void
     */
    public function test_creator_can_view_provider_dashboard()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Create a creator user
        $user = User::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        $user->assignRole('creator');
        
        // Acting as creator
        $this->actingAs($user);
        
        // Create social accounts
        SocialAccount::factory()->create([
            'user_id' => $user->id,
            'provider' => 'youtube',
            'meta' => ['followers' => 1000],
        ]);
        
        // Visit provider dashboard
        $response = $this->get(route('creator.provider.show', ['provider' => 'youtube']));
        
        // Should see provider dashboard
        $response->assertStatus(200);
        $response->assertSee('YouTube');
    }

    /**
     * Test creator dashboard with no social accounts.
     *
     * @return void
     */
    public function test_creator_dashboard_with_no_social_accounts()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Create a creator user
        $user = User::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        $user->assignRole('creator');
        
        // Acting as creator
        $this->actingAs($user);
        
        // Visit dashboard
        $response = $this->get(route('creator.dashboard'));
        
        // Should see dashboard with no social accounts message
        $response->assertStatus(200);
        $response->assertSee('No connections');
    }

    /**
     * Test unauthorized access to another creator's social account.
     *
     * @return void
     */
    public function test_unauthorized_access_to_another_creators_social_account()
    {
        // Create a tenant
        $tenant = \App\Models\Tenant::firstOrCreate(
            ['domain' => 'demo.local'],
            [
                'name' => 'DemoTenant',
                'plan_id' => null,
                'billing_info' => null,
                'active' => true,
            ]
        );
        
        // Create two creators
        $creator1 = User::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        $creator1->assignRole('creator');
        
        $creator2 = User::factory()->create([
            'tenant_id' => $tenant->id,
        ]);
        $creator2->assignRole('creator');
        
        // Create a social account for creator1
        $account = SocialAccount::factory()->create([
            'user_id' => $creator1->id,
            'provider' => 'youtube',
        ]);
        
        // Acting as creator2
        $this->actingAs($creator2);
        
        // Try to access creator1's account
        $response = $this->get(route('creator.provider.show', ['provider' => 'youtube']));
        
        // Should get successful response but not see creator1's account
        $response->assertStatus(200);
    }
}