<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Brand;
use App\Models\Tenant;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class NavigationAndRedirectsTest extends TestCase
{
    use RefreshDatabase;

    protected $brandAdmin;
    protected $creator;
    protected $tenant;
    protected $brand;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->exists()) {
            Role::create(['name' => 'brand_admin']);
        }
        
        if (!Role::where('name', 'creator')->exists()) {
            Role::create(['name' => 'creator']);
        }

        // Create a tenant
        $this->tenant = Tenant::factory()->create();

        // Create a brand
        $this->brand = Brand::factory()->create([
            'tenant_id' => $this->tenant->id,
        ]);

        // Create a brand admin user
        $this->brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
            'tenant_id' => $this->tenant->id,
            'brand_id' => $this->brand->id,
        ]);
        $this->brandAdmin->assignRole('brand_admin');

        // Create a creator user
        $this->creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
            'tenant_id' => $this->tenant->id,
        ]);
        $this->creator->assignRole('creator');
    }

    public function test_dashboard_redirect_for_brand_admin()
    {
        // Login as brand admin
        $response = $this->actingAs($this->brandAdmin)
            ->get('/dashboard');
        
        // Should redirect to brand dashboard
        $response->assertRedirect(route('brand.dashboard'));
    }

    public function test_dashboard_redirect_for_creator()
    {
        // Login as creator
        $response = $this->actingAs($this->creator)
            ->get('/dashboard');
        
        // Should redirect to creator dashboard
        $response->assertRedirect(route('creator.dashboard'));
    }

    public function test_brand_dashboard_access()
    {
        // Login as brand admin
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.dashboard'));
        
        // Should be successful
        $response->assertStatus(200);
    }

    public function test_creator_dashboard_access()
    {
        // Login as creator
        $response = $this->actingAs($this->creator)
            ->get(route('creator.dashboard'));
        
        // Should be successful
        $response->assertStatus(200);
    }

    public function test_brand_navigation_links()
    {
        // Login as brand admin
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.dashboard'));
        
        // Check that navigation links are present
        $response->assertSee('Dashboard');
        $response->assertSee('Connections');
        $response->assertSee('Creators');
        $response->assertSee('Proposals');
        $response->assertSee('Reels');
        $response->assertSee('Messages');
    }

    public function test_creator_navigation_links()
    {
        // Login as creator
        $response = $this->actingAs($this->creator)
            ->get(route('creator.dashboard'));
        
        // Check that navigation links are present
        $response->assertSee('Dashboard');
        $response->assertSee('Connections');
        $response->assertSee('Content');
        $response->assertSee('Proposals');
        $response->assertSee('Reels');
        $response->assertSee('Messages');
    }

    public function test_unauthorized_access_to_brand_routes()
    {
        // Login as creator
        $response = $this->actingAs($this->creator)
            ->get(route('brand.dashboard'));
        
        // Should be forbidden
        $response->assertStatus(403);
    }

    public function test_unauthorized_access_to_creator_routes()
    {
        // Login as brand admin
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('creator.dashboard'));
        
        // Should be forbidden
        $response->assertStatus(403);
    }
}