<?php

namespace Tests\Feature;

use App\Jobs\ProcessPerformanceForecast;
use App\Models\Brand;
use App\Models\Campaign;
use App\Models\PerformanceForecast;
use App\Models\SocialAccount;
use App\Models\SocialPost;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Queue;
use Tests\TestCase;

class PerformanceForecastTest extends TestCase
{
    use RefreshDatabase;

    protected $brandAdmin;
    protected $brand;

    protected function setUp(): void
    {
        parent::setUp();

        // Create brand and brand admin user
        $this->brand = Brand::factory()->create();
        $this->brandAdmin = User::factory()->create([
            'brand_id' => $this->brand->id,
        ]);
        $this->brandAdmin->assignRole('brand_admin');
    }

    /** @test */
    public function brand_admin_can_view_forecasts_index()
    {
        $this->actingAs($this->brandAdmin);

        $response = $this->get(route('brand.forecasts.index'));

        $response->assertStatus(200);
        $response->assertViewIs('brand.forecasts.index');
    }

    /** @test */
    public function brand_admin_can_view_create_forecast_form()
    {
        $this->actingAs($this->brandAdmin);

        $response = $this->get(route('brand.forecasts.create'));

        $response->assertStatus(200);
        $response->assertViewIs('brand.forecasts.create');
    }

    /** @test */
    public function brand_admin_can_create_forecast()
    {
        Queue::fake();

        $this->actingAs($this->brandAdmin);

        $campaign = Campaign::factory()->create(['brand_id' => $this->brand->id]);
        $socialAccount = SocialAccount::factory()->create(['brand_id' => $this->brand->id]);

        $data = [
            'start_date' => now()->subDays(60)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
            'forecast_days' => 30,
            'scenario' => 'balanced',
            'campaign_ids' => [$campaign->id],
            'social_account_ids' => [$socialAccount->id],
        ];

        $response = $this->post(route('brand.forecasts.store'), $data);

        $response->assertRedirect();
        $this->assertDatabaseHas('performance_forecasts', [
            'brand_id' => $this->brand->id,
            'forecast_days' => 30,
            'scenario' => 'balanced',
            'status' => 'pending',
        ]);

        Queue::assertPushed(ProcessPerformanceForecast::class);
    }

    /** @test */
    public function forecast_requires_valid_date_range()
    {
        $this->actingAs($this->brandAdmin);

        $data = [
            'start_date' => now()->format('Y-m-d'),
            'end_date' => now()->subDays(30)->format('Y-m-d'), // End before start
            'forecast_days' => 30,
            'scenario' => 'balanced',
        ];

        $response = $this->post(route('brand.forecasts.store'), $data);

        $response->assertSessionHasErrors('end_date');
    }

    /** @test */
    public function forecast_requires_valid_scenario()
    {
        $this->actingAs($this->brandAdmin);

        $data = [
            'start_date' => now()->subDays(60)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
            'forecast_days' => 30,
            'scenario' => 'invalid_scenario',
        ];

        $response = $this->post(route('brand.forecasts.store'), $data);

        $response->assertSessionHasErrors('scenario');
    }

    /** @test */
    public function brand_admin_can_view_own_forecast()
    {
        $this->actingAs($this->brandAdmin);

        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'created_by' => $this->brandAdmin->id,
        ]);

        $response = $this->get(route('brand.forecasts.show', $forecast));

        $response->assertStatus(200);
        $response->assertViewIs('brand.forecasts.show');
        $response->assertViewHas('forecast', $forecast);
    }

    /** @test */
    public function brand_admin_cannot_view_other_brands_forecast()
    {
        $this->actingAs($this->brandAdmin);

        $otherBrand = Brand::factory()->create();
        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $otherBrand->id,
        ]);

        $response = $this->get(route('brand.forecasts.show', $forecast));

        $response->assertStatus(403);
    }

    /** @test */
    public function brand_admin_can_delete_own_forecast()
    {
        $this->actingAs($this->brandAdmin);

        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'created_by' => $this->brandAdmin->id,
        ]);

        $response = $this->delete(route('brand.forecasts.destroy', $forecast));

        $response->assertRedirect(route('brand.forecasts.index'));
        $this->assertDatabaseMissing('performance_forecasts', [
            'id' => $forecast->id,
        ]);
    }

    /** @test */
    public function brand_admin_can_get_forecast_data_json()
    {
        $this->actingAs($this->brandAdmin);

        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'created_by' => $this->brandAdmin->id,
            'status' => 'completed',
            'predicted_timeseries' => [
                'dates' => ['2025-01-01', '2025-01-02'],
                'impressions' => [1000, 1100],
                'engagement' => [50, 55],
                'conversions' => [10, 12],
            ],
            'error_bands' => [
                'impressions' => ['lower' => [900, 1000], 'upper' => [1100, 1200]],
            ],
            'roi_estimates' => [
                'roi' => 15.5,
                'reach' => 50000,
                'conversions' => 500,
                'engagement_rate' => 3.2,
            ],
        ]);

        $response = $this->get(route('brand.forecasts.data', $forecast));

        $response->assertStatus(200);
        $response->assertJson([
            'status' => 'completed',
            'predicted_timeseries' => $forecast->predicted_timeseries,
            'roi_estimates' => $forecast->roi_estimates,
        ]);
    }

    /** @test */
    public function forecast_data_returns_processing_status_for_incomplete_forecast()
    {
        $this->actingAs($this->brandAdmin);

        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'created_by' => $this->brandAdmin->id,
            'status' => 'processing',
        ]);

        $response = $this->get(route('brand.forecasts.data', $forecast));

        $response->assertStatus(202);
        $response->assertJson([
            'status' => 'processing',
            'message' => 'Forecast is still processing',
        ]);
    }

    /** @test */
    public function brand_admin_can_export_completed_forecast()
    {
        $this->actingAs($this->brandAdmin);

        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'created_by' => $this->brandAdmin->id,
            'status' => 'completed',
            'roi_estimates' => ['roi' => 15.5],
        ]);

        $response = $this->get(route('brand.forecasts.export', $forecast));

        $response->assertStatus(200);
        $response->assertHeader('Content-Type', 'application/json');
    }

    /** @test */
    public function forecast_model_has_correct_relationships()
    {
        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'created_by' => $this->brandAdmin->id,
        ]);

        $this->assertInstanceOf(Brand::class, $forecast->brand);
        $this->assertInstanceOf(User::class, $forecast->creator);
        $this->assertEquals($this->brand->id, $forecast->brand->id);
        $this->assertEquals($this->brandAdmin->id, $forecast->creator->id);
    }

    /** @test */
    public function forecast_can_be_marked_as_completed()
    {
        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'status' => 'processing',
        ]);

        $results = [
            'predicted_timeseries' => ['dates' => [], 'impressions' => []],
            'error_bands' => [],
            'roi_estimates' => ['roi' => 10],
            'insights_text' => 'Test insights',
            'insights_meta' => [],
        ];

        $forecast->markAsCompleted($results);

        $this->assertEquals('completed', $forecast->status);
        $this->assertNotNull($forecast->completed_at);
        $this->assertEquals(10, $forecast->roi_estimates['roi']);
    }

    /** @test */
    public function forecast_can_be_marked_as_failed()
    {
        $forecast = PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'status' => 'processing',
        ]);

        $forecast->markAsFailed('Test error message');

        $this->assertEquals('failed', $forecast->status);
        $this->assertEquals('Test error message', $forecast->error_message);
    }

    /** @test */
    public function forecast_scopes_work_correctly()
    {
        PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'status' => 'completed',
        ]);

        PerformanceForecast::factory()->create([
            'brand_id' => $this->brand->id,
            'status' => 'pending',
        ]);

        $otherBrand = Brand::factory()->create();
        PerformanceForecast::factory()->create([
            'brand_id' => $otherBrand->id,
            'status' => 'completed',
        ]);

        $brandForecasts = PerformanceForecast::forBrand($this->brand->id)->get();
        $completedForecasts = PerformanceForecast::completed()->get();
        $pendingForecasts = PerformanceForecast::pending()->get();

        $this->assertCount(2, $brandForecasts);
        $this->assertCount(2, $completedForecasts);
        $this->assertCount(1, $pendingForecasts);
    }

    /** @test */
    public function non_brand_admin_cannot_access_forecasts()
    {
        $regularUser = User::factory()->create();

        $this->actingAs($regularUser);

        $response = $this->get(route('brand.forecasts.index'));

        $response->assertStatus(403);
    }

    /** @test */
    public function forecast_validates_campaign_belongs_to_brand()
    {
        $this->actingAs($this->brandAdmin);

        $otherBrand = Brand::factory()->create();
        $otherCampaign = Campaign::factory()->create(['brand_id' => $otherBrand->id]);

        $data = [
            'start_date' => now()->subDays(60)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
            'forecast_days' => 30,
            'scenario' => 'balanced',
            'campaign_ids' => [$otherCampaign->id],
        ];

        $response = $this->postJson(route('brand.forecasts.store'), $data);

        $response->assertStatus(403);
    }
}