<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Brand;
use App\Models\ReputationScore;
use App\Models\ReputationDriver;
use App\Models\ReputationRecommendation;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ReputationScoreTest extends TestCase
{
    use RefreshDatabase;

    protected User $brandUser;
    protected Brand $brand;

    protected function setUp(): void
    {
        parent::setUp();

        $this->brand = Brand::factory()->create();
        $this->brandUser = User::factory()->create([
            'role' => 'brand',
            'brand_id' => $this->brand->id,
        ]);
    }

    /** @test */
    public function brand_admin_can_view_reputation_dashboard()
    {
        $response = $this->actingAs($this->brandUser)
            ->get(route('brand.reputation.index'));

        $response->assertStatus(200);
        $response->assertViewIs('brand.reputation.index');
    }

    /** @test */
    public function brand_admin_can_calculate_reputation_score()
    {
        $response = $this->actingAs($this->brandUser)
            ->post(route('brand.reputation.calculate'), [
                'timeframe' => '30',
            ]);

        $response->assertRedirect();
        $this->assertDatabaseHas('reputation_scores', [
            'brand_id' => $this->brand->id,
            'timeframe' => '30',
        ]);
    }

    /** @test */
    public function reputation_score_calculates_all_subscores()
    {
        $score = ReputationScore::factory()->create([
            'brand_id' => $this->brand->id,
            'score' => 75.5,
            'sentiment_score' => 80.0,
            'responsiveness_score' => 70.0,
            'trust_score' => 75.0,
            'compliance_score' => 85.0,
            'growth_score' => 65.0,
        ]);

        $this->assertEquals(75.5, $score->score);
        $this->assertEquals('B', $score->grade);
        $this->assertEquals('good', $score->status_label);
    }

    /** @test */
    public function brand_admin_can_resolve_negative_driver()
    {
        $score = ReputationScore::factory()->create(['brand_id' => $this->brand->id]);
        $driver = ReputationDriver::factory()->create([
            'reputation_score_id' => $score->id,
            'brand_id' => $this->brand->id,
            'type' => 'negative',
            'is_resolved' => false,
        ]);

        $response = $this->actingAs($this->brandUser)
            ->post(route('brand.reputation.drivers.resolve', $driver), [
                'resolution_note' => 'Issue has been addressed',
            ]);

        $response->assertRedirect();
        $this->assertDatabaseHas('reputation_drivers', [
            'id' => $driver->id,
            'is_resolved' => true,
        ]);
    }

    /** @test */
    public function brand_admin_can_update_recommendation_status()
    {
        $score = ReputationScore::factory()->create(['brand_id' => $this->brand->id]);
        $recommendation = ReputationRecommendation::factory()->create([
            'reputation_score_id' => $score->id,
            'brand_id' => $this->brand->id,
            'status' => 'pending',
        ]);

        $response = $this->actingAs($this->brandUser)
            ->put(route('brand.reputation.recommendations.update', $recommendation), [
                'status' => 'in_progress',
            ]);

        $response->assertRedirect();
        $this->assertDatabaseHas('reputation_recommendations', [
            'id' => $recommendation->id,
            'status' => 'in_progress',
        ]);
    }

    /** @test */
    public function score_detects_significant_drop_and_triggers_alert()
    {
        $newScore = ReputationScore::factory()->create([
            'brand_id' => $this->brand->id,
            'score' => 65.0,
            'previous_score' => 80.0,
            'score_change' => -15.0,
            'score_change_percentage' => -18.75,
            'alert_triggered' => true,
        ]);

        $this->assertTrue($newScore->alert_triggered);
        $this->assertTrue($newScore->hasSignificantDrop());
    }

    /** @test */
    public function brand_cannot_view_other_brands_reputation()
    {
        $otherBrand = Brand::factory()->create();
        $otherScore = ReputationScore::factory()->create(['brand_id' => $otherBrand->id]);

        $response = $this->actingAs($this->brandUser)
            ->get(route('brand.reputation.show', $otherScore));

        $response->assertStatus(403);
    }

    /** @test */
    public function drivers_are_categorized_correctly()
    {
        $score = ReputationScore::factory()->create(['brand_id' => $this->brand->id]);
        
        $positiveDriver = ReputationDriver::factory()->create([
            'reputation_score_id' => $score->id,
            'brand_id' => $this->brand->id,
            'type' => 'positive',
            'impact_score' => 5.5,
        ]);

        $negativeDriver = ReputationDriver::factory()->create([
            'reputation_score_id' => $score->id,
            'brand_id' => $this->brand->id,
            'type' => 'negative',
            'impact_score' => -4.2,
        ]);

        $this->assertEquals('high', $positiveDriver->impact_level);
        $this->assertEquals('medium', $negativeDriver->impact_level);
    }

    /** @test */
    public function audit_log_tracks_score_calculations()
    {
        $score = ReputationScore::factory()->create([
            'brand_id' => $this->brand->id,
            'status' => 'completed',
        ]);

        $this->assertDatabaseHas('reputation_audit_logs', [
            'brand_id' => $this->brand->id,
            'reputation_score_id' => $score->id,
            'action' => 'score_calculated',
        ]);
    }

    /** @test */
    public function recommendations_are_prioritized_by_impact()
    {
        $score = ReputationScore::factory()->create(['brand_id' => $this->brand->id]);
        
        $highPriority = ReputationRecommendation::factory()->create([
            'reputation_score_id' => $score->id,
            'brand_id' => $this->brand->id,
            'priority' => 'high',
            'estimated_impact' => 20,
        ]);

        $lowPriority = ReputationRecommendation::factory()->create([
            'reputation_score_id' => $score->id,
            'brand_id' => $this->brand->id,
            'priority' => 'low',
            'estimated_impact' => 5,
        ]);

        $recommendations = $score->recommendations()->orderByPriority()->get();
        
        $this->assertEquals($highPriority->id, $recommendations->first()->id);
    }
}