
# AI Microservice - FastAPI + Celery

A comprehensive Python microservice for AI tasks, social media publishing, and automation for the "AI Business Market" Laravel platform.

## 🚀 Features

- **AI Task Processing**: LLM calls, content generation, proposals, translations, forecasting
- **Social Media Publishing**: Twitter, Facebook, Instagram, LinkedIn, YouTube
- **RAG (Retrieval-Augmented Generation)**: Vector store with FAISS/Weaviate
- **Reel Processing**: Video processing orchestration
- **Async Task Queue**: Celery with Redis
- **Security**: PII redaction, encryption, secure token management
- **Observability**: Prometheus metrics, structured logging, cost tracking
- **Idempotency**: Built-in support for idempotent operations
- **Testing**: Comprehensive test suite with mock providers

## 📋 Prerequisites

- Python 3.11+
- PostgreSQL 15+
- Redis 7+
- Docker & Docker Compose (recommended)

## 🛠️ Installation

### Using Docker Compose (Recommended)

1. Clone the repository:
```bash
git clone <repository-url>
cd ai-microservice
```

2. Copy environment file:
```bash
cp .env.example .env
```

3. Update `.env` with your credentials:
```bash
# Required: Add your API keys
OPENAI_API_KEY=sk-your-key
ANTHROPIC_API_KEY=sk-ant-your-key
LARAVEL_API_KEY=your-laravel-key
SECRET_KEY=your-secret-key
ENCRYPTION_KEY=your-32-byte-encryption-key
```

4. Start services:
```bash
# Start core services
docker-compose up -d

# Start with monitoring (Prometheus, Grafana, Flower)
docker-compose --profile monitoring up -d

# Start with Weaviate vector store
docker-compose --profile weaviate up -d
```

5. Initialize database:
```bash
docker-compose exec api python -c "from app.database import init_db; import asyncio; asyncio.run(init_db())"
```

6. Access services:
- API: http://localhost:8000
- API Docs: http://localhost:8000/docs
- Flower (Celery monitoring): http://localhost:5555
- Grafana: http://localhost:3000 (admin/admin)
- Prometheus: http://localhost:9090

### Manual Installation

1. Create virtual environment:
```bash
python -m venv venv
source venv/bin/activate  # On Windows: venv\Scripts\activate
```

2. Install dependencies:
```bash
pip install -r requirements.txt
```

3. Set up environment:
```bash
cp .env.example .env
# Edit .env with your configuration
```

4. Start services:
```bash
# Terminal 1: Start API
uvicorn app.main:app --reload

# Terminal 2: Start Celery worker
celery -A app.celery_app worker --loglevel=info

# Terminal 3: Start Celery beat (scheduler)
celery -A app.celery_app beat --loglevel=info
```

## 📚 API Documentation

### Authentication

All endpoints require API key authentication via header:
```
X-API-Key: your-laravel-api-key
```

Admin endpoints require:
```
X-Admin-Key: your-admin-key
```

### Core Endpoints

#### 1. Create AI Job
```http
POST /v1/ai/jobs
Content-Type: application/json
X-API-Key: your-key

{
  "job_type": "content_ideas",
  "tenant_id": "brand-123",
  "owner_id": "user-456",
  "payload": {
    "topic": "AI in marketing",
    "count": 5,
    "tone": "professional"
  },
  "callback_url": "https://laravel.example/api/ai-callback",
  "idempotency_key": "unique-key-123"
}
```

**Response:**
```json
{
  "ai_job_id": "job-789",
  "status": "pending",
  "estimated_time_seconds": 30,
  "message": "Job queued successfully"
}
```

#### 2. Get Job Status
```http
GET /v1/ai/jobs/{job_id}
X-API-Key: your-key
```

#### 3. Publish to Social Media
```http
POST /v1/publish
Content-Type: application/json
X-API-Key: your-key

{
  "platform": "twitter",
  "tenant_id": "brand-123",
  "owner_id": "user-456",
  "content": {
    "text": "Check out our new product!",
    "media_urls": ["https://example.com/image.jpg"]
  },
  "callback_url": "https://laravel.example/api/publish-callback"
}
```

#### 4. RAG - Ingest Documents
```http
POST /v1/rag/ingest
Content-Type: application/json
X-API-Key: your-key

{
  "tenant_id": "brand-123",
  "documents": [
    {
      "id": "doc-1",
      "title": "Product FAQ",
      "content": "Q: What is the product? A: ...",
      "metadata": {"category": "faq"}
    }
  ],
  "document_type": "support_doc"
}
```

#### 5. RAG - Query
```http
POST /v1/rag/query
Content-Type: application/json
X-API-Key: your-key

{
  "tenant_id": "brand-123",
  "query": "How do I reset my password?",
  "top_k": 5,
  "include_sources": true
}
```

### Job Types

- `proposal`: Generate business proposals
- `content_ideas`: Generate content ideas
- `auto_reply`: Generate auto-replies
- `ads_generate`: Generate ad variants
- `forecast`: Generate forecasts/predictions
- `matchmaking`: Match influencers with brands
- `translate`: Translate content
- `contract_draft`: Draft contracts
- `support_rag`: Support queries using RAG
- `custom`: Custom prompts

### Social Platforms

- `twitter`: Twitter/X
- `facebook`: Facebook Pages
- `instagram`: Instagram Business
- `linkedin`: LinkedIn
- `youtube`: YouTube (coming soon)

## 🔧 Configuration

### Environment Variables

Key configuration options in `.env`:

```bash
# Application
APP_ENV=production
APP_DEBUG=false
LOG_LEVEL=INFO

# AI Providers
OPENAI_API_KEY=sk-your-key
OPENAI_DEFAULT_MODEL=gpt-4-turbo-preview
ANTHROPIC_API_KEY=sk-ant-your-key
LOCAL_LLM_ENABLED=false

# Vector Store
VECTOR_STORE_TYPE=faiss  # faiss, weaviate, pgvector
EMBEDDING_MODEL=sentence-transformers/all-MiniLM-L6-v2

# Security
PII_REDACTION_ENABLED=true
ENCRYPTION_KEY=your-32-byte-key

# Observability
ENABLE_METRICS=true
TRACK_COSTS=true

# Testing
TEST_MODE=false
MOCK_PROVIDERS=false
```

## 🧪 Testing

Run tests:
```bash
# All tests
pytest

# With coverage
pytest --cov=app --cov-report=html

# Specific test file
pytest tests/test_api.py

# Verbose output
pytest -v
```

## 📊 Monitoring

### Metrics

Prometheus metrics available at `/metrics`:
- Job counts by type and status
- LLM token usage and costs
- API latency
- Worker status

### Logging

Structured JSON logs with:
- Request/response tracking
- Error details
- Performance metrics
- Cost tracking

### Cost Tracking

Automatic cost calculation for:
- OpenAI API calls
- Anthropic API calls
- Token usage tracking
- Per-tenant cost breakdown

## 🔒 Security

### PII Protection

Automatic PII redaction for:
- Email addresses
- Phone numbers
- Credit card numbers
- IP addresses
- Custom patterns

### Encryption

- Sensitive data encrypted at rest
- Secure token storage
- HMAC signature validation for callbacks

### Rate Limiting

Configurable rate limits:
- Per minute: 60 requests
- Per hour: 1000 requests

## 🚀 Deployment

### Production Checklist

1. ✅ Set `APP_ENV=production`
2. ✅ Set `APP_DEBUG=false`
3. ✅ Use strong `SECRET_KEY` and `ENCRYPTION_KEY`
4. ✅ Configure production database
5. ✅ Set up SSL/TLS
6. ✅ Configure firewall rules
7. ✅ Set up monitoring and alerts
8. ✅ Configure backup strategy
9. ✅ Review and set rate limits
10. ✅ Configure log aggregation

### Kubernetes Deployment

Example deployment files in `k8s/`