"""
Reel/Video Processing router
Orchestrates video processing with external processor
"""
from fastapi import APIRouter, Depends, HTTPException, status, Header
from typing import Optional
import httpx

from app.schemas import ReelProcessRequest, ReelProcessResponse
from app.utils.observability import logger
from config import settings

router = APIRouter()


async def verify_api_key(x_api_key: Optional[str] = Header(None)):
    """Verify API key from Laravel"""
    if not x_api_key or x_api_key != settings.laravel_api_key:
        raise HTTPException(
            status_code=status.HTTP_401_UNAUTHORIZED,
            detail="Invalid or missing API key"
        )
    return x_api_key


@router.post("/process", response_model=ReelProcessResponse)
async def process_reel(
    request: ReelProcessRequest,
    api_key: str = Depends(verify_api_key)
):
    """
    Process a reel/video
    
    Forwards request to external reel processor and returns job ID.
    The processor will callback to Laravel when complete.
    """
    try:
        # Forward to external reel processor
        async with httpx.AsyncClient(timeout=30.0) as client:
            headers = {}
            if settings.reel_processor_api_key:
                headers["Authorization"] = f"Bearer {settings.reel_processor_api_key}"
            
            response = await client.post(
                f"{settings.reel_processor_url}/process",
                json={
                    "video_url": request.video_url,
                    "options": request.processing_options,
                    "callback_url": request.callback_url or settings.laravel_base_url + "/api/reel-callback",
                    "tenant_id": request.tenant_id,
                    "owner_id": request.owner_id,
                    "idempotency_key": request.idempotency_key
                },
                headers=headers
            )
            
            response.raise_for_status()
            data = response.json()
        
        logger.info(
            "Reel processing job created",
            reel_job_id=data.get("job_id"),
            tenant_id=request.tenant_id
        )
        
        return ReelProcessResponse(
            reel_job_id=data.get("job_id", "unknown"),
            status=data.get("status", "pending"),
            message=data.get("message", "Reel processing started"),
            estimated_time_seconds=data.get("estimated_time_seconds")
        )
        
    except httpx.HTTPStatusError as e:
        logger.error(
            "Reel processor returned error",
            status_code=e.response.status_code,
            error=e.response.text
        )
        raise HTTPException(
            status_code=status.HTTP_502_BAD_GATEWAY,
            detail=f"Reel processor error: {e.response.text}"
        )
    except httpx.RequestError as e:
        logger.error("Failed to connect to reel processor", error=str(e))
        raise HTTPException(
            status_code=status.HTTP_503_SERVICE_UNAVAILABLE,
            detail="Reel processor unavailable"
        )
    except Exception as e:
        logger.error("Failed to process reel", error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to process reel: {str(e)}"
        )


@router.post("/callback")
async def reel_callback(
    payload: dict,
    x_signature: Optional[str] = Header(None)
):
    """
    Receive callback from reel processor
    
    Validates signature and forwards to Laravel.
    """
    try:
        # Validate signature if provided
        if x_signature and settings.reel_processor_api_key:
            from app.utils.security import SignatureValidator
            import json
            
            payload_str = json.dumps(payload, sort_keys=True)
            if not SignatureValidator.validate_signature(
                payload_str,
                x_signature,
                settings.reel_processor_api_key
            ):
                raise HTTPException(
                    status_code=status.HTTP_401_UNAUTHORIZED,
                    detail="Invalid signature"
                )
        
        logger.info(
            "Received reel callback",
            job_id=payload.get("job_id"),
            status=payload.get("status")
        )
        
        # Forward to Laravel
        async with httpx.AsyncClient(timeout=30.0) as client:
            response = await client.post(
                f"{settings.laravel_base_url}/api/reel-callback",
                json=payload,
                headers={"Authorization": f"Bearer {settings.laravel_api_key}"}
            )
            response.raise_for_status()
        
        logger.info("Reel callback forwarded to Laravel", job_id=payload.get("job_id"))
        
        return {"message": "Callback processed successfully"}
        
    except httpx.HTTPStatusError as e:
        logger.error(
            "Laravel callback failed",
            status_code=e.response.status_code,
            error=e.response.text
        )
        raise HTTPException(
            status_code=status.HTTP_502_BAD_GATEWAY,
            detail="Failed to forward callback to Laravel"
        )
    except Exception as e:
        logger.error("Failed to process reel callback", error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to process callback: {str(e)}"
        )


@router.get("/status/{job_id}")
async def get_reel_status(
    job_id: str,
    api_key: str = Depends(verify_api_key)
):
    """
    Get status of a reel processing job
    
    Queries external processor for job status.
    """
    try:
        async with httpx.AsyncClient(timeout=10.0) as client:
            headers = {}
            if settings.reel_processor_api_key:
                headers["Authorization"] = f"Bearer {settings.reel_processor_api_key}"
            
            response = await client.get(
                f"{settings.reel_processor_url}/status/{job_id}",
                headers=headers
            )
            response.raise_for_status()
            return response.json()
        
    except httpx.HTTPStatusError as e:
        if e.response.status_code == 404:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail=f"Reel job {job_id} not found"
            )
        raise HTTPException(
            status_code=status.HTTP_502_BAD_GATEWAY,
            detail="Failed to get reel status"
        )
    except Exception as e:
        logger.error("Failed to get reel status", job_id=job_id, error=str(e))
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Failed to get reel status: {str(e)}"
        )