"""
Configuration management for AI Microservice
Loads settings from environment variables with validation
"""
from typing import Optional, List
from pydantic_settings import BaseSettings
from pydantic import Field, validator
import os


class Settings(BaseSettings):
    """Application settings with environment variable support"""
    
    # Application
    app_name: str = Field(default="ai-microservice", env="APP_NAME")
    app_env: str = Field(default="development", env="APP_ENV")
    app_debug: bool = Field(default=True, env="APP_DEBUG")
    app_host: str = Field(default="0.0.0.0", env="APP_HOST")
    app_port: int = Field(default=8000, env="APP_PORT")
    log_level: str = Field(default="INFO", env="LOG_LEVEL")
    
    # Security
    secret_key: str = Field(..., env="SECRET_KEY")
    encryption_key: str = Field(..., env="ENCRYPTION_KEY")
    jwt_secret: str = Field(..., env="JWT_SECRET")
    allowed_origins: List[str] = Field(
        default=["http://localhost:3000"],
        env="ALLOWED_ORIGINS"
    )
    
    # Redis
    redis_host: str = Field(default="localhost", env="REDIS_HOST")
    redis_port: int = Field(default=6379, env="REDIS_PORT")
    redis_db: int = Field(default=0, env="REDIS_DB")
    redis_password: Optional[str] = Field(default=None, env="REDIS_PASSWORD")
    redis_url: str = Field(default="redis://localhost:6379/0", env="REDIS_URL")
    
    # Celery
    celery_broker_url: str = Field(
        default="redis://localhost:6379/0",
        env="CELERY_BROKER_URL"
    )
    celery_result_backend: str = Field(
        default="redis://localhost:6379/1",
        env="CELERY_RESULT_BACKEND"
    )
    celery_task_track_started: bool = Field(default=True, env="CELERY_TASK_TRACK_STARTED")
    celery_task_time_limit: int = Field(default=3600, env="CELERY_TASK_TIME_LIMIT")
    celery_worker_prefetch_multiplier: int = Field(
        default=4,
        env="CELERY_WORKER_PREFETCH_MULTIPLIER"
    )
    
    # Database
    database_url: str = Field(
        default="postgresql+asyncpg://user:password@localhost:5432/ai_microservice",
        env="DATABASE_URL"
    )
    db_pool_size: int = Field(default=20, env="DB_POOL_SIZE")
    db_max_overflow: int = Field(default=10, env="DB_MAX_OVERFLOW")
    
    # S3-Compatible Storage
    s3_endpoint_url: str = Field(default="http://localhost:9000", env="S3_ENDPOINT_URL")
    s3_access_key: str = Field(default="minioadmin", env="S3_ACCESS_KEY")
    s3_secret_key: str = Field(default="minioadmin", env="S3_SECRET_KEY")
    s3_bucket_name: str = Field(default="ai-microservice", env="S3_BUCKET_NAME")
    s3_region: str = Field(default="us-east-1", env="S3_REGION")
    s3_use_ssl: bool = Field(default=False, env="S3_USE_SSL")
    
    # Vector Store
    vector_store_type: str = Field(default="faiss", env="VECTOR_STORE_TYPE")
    weaviate_url: Optional[str] = Field(default=None, env="WEAVIATE_URL")
    weaviate_api_key: Optional[str] = Field(default=None, env="WEAVIATE_API_KEY")
    faiss_index_path: str = Field(default="./data/faiss_index", env="FAISS_INDEX_PATH")
    embedding_model: str = Field(
        default="sentence-transformers/all-MiniLM-L6-v2",
        env="EMBEDDING_MODEL"
    )
    embedding_dimension: int = Field(default=384, env="EMBEDDING_DIMENSION")
    
    # OpenAI
    openai_api_key: Optional[str] = Field(default=None, env="OPENAI_API_KEY")
    openai_org_id: Optional[str] = Field(default=None, env="OPENAI_ORG_ID")
    openai_default_model: str = Field(
        default="gpt-4-turbo-preview",
        env="OPENAI_DEFAULT_MODEL"
    )
    openai_embedding_model: str = Field(
        default="text-embedding-3-small",
        env="OPENAI_EMBEDDING_MODEL"
    )
    openai_max_tokens: int = Field(default=4096, env="OPENAI_MAX_TOKENS")
    openai_temperature: float = Field(default=0.7, env="OPENAI_TEMPERATURE")
    
    # Anthropic
    anthropic_api_key: Optional[str] = Field(default=None, env="ANTHROPIC_API_KEY")
    anthropic_default_model: str = Field(
        default="claude-3-sonnet-20240229",
        env="ANTHROPIC_DEFAULT_MODEL"
    )
    anthropic_max_tokens: int = Field(default=4096, env="ANTHROPIC_MAX_TOKENS")
    
    # Local LLM
    local_llm_enabled: bool = Field(default=False, env="LOCAL_LLM_ENABLED")
    local_llm_url: str = Field(default="http://localhost:11434", env="LOCAL_LLM_URL")
    local_llm_model: str = Field(default="llama2", env="LOCAL_LLM_MODEL")
    
    # Social Media - Twitter
    twitter_api_key: Optional[str] = Field(default=None, env="TWITTER_API_KEY")
    twitter_api_secret: Optional[str] = Field(default=None, env="TWITTER_API_SECRET")
    twitter_access_token: Optional[str] = Field(default=None, env="TWITTER_ACCESS_TOKEN")
    twitter_access_secret: Optional[str] = Field(default=None, env="TWITTER_ACCESS_SECRET")
    twitter_bearer_token: Optional[str] = Field(default=None, env="TWITTER_BEARER_TOKEN")
    
    # Social Media - Facebook
    facebook_app_id: Optional[str] = Field(default=None, env="FACEBOOK_APP_ID")
    facebook_app_secret: Optional[str] = Field(default=None, env="FACEBOOK_APP_SECRET")
    facebook_access_token: Optional[str] = Field(default=None, env="FACEBOOK_ACCESS_TOKEN")
    
    # Social Media - YouTube
    youtube_client_id: Optional[str] = Field(default=None, env="YOUTUBE_CLIENT_ID")
    youtube_client_secret: Optional[str] = Field(default=None, env="YOUTUBE_CLIENT_SECRET")
    youtube_refresh_token: Optional[str] = Field(default=None, env="YOUTUBE_REFRESH_TOKEN")
    
    # Social Media - LinkedIn
    linkedin_client_id: Optional[str] = Field(default=None, env="LINKEDIN_CLIENT_ID")
    linkedin_client_secret: Optional[str] = Field(default=None, env="LINKEDIN_CLIENT_SECRET")
    linkedin_access_token: Optional[str] = Field(default=None, env="LINKEDIN_ACCESS_TOKEN")
    
    # Laravel Integration
    laravel_base_url: str = Field(
        default="https://laravel.example.com",
        env="LARAVEL_BASE_URL"
    )
    laravel_api_key: str = Field(..., env="LARAVEL_API_KEY")
    laravel_callback_secret: str = Field(..., env="LARAVEL_CALLBACK_SECRET")
    
    # Reel Processing
    reel_processor_url: str = Field(
        default="http://localhost:8001",
        env="REEL_PROCESSOR_URL"
    )
    reel_processor_api_key: Optional[str] = Field(
        default=None,
        env="REEL_PROCESSOR_API_KEY"
    )
    reel_processor_timeout: int = Field(default=300, env="REEL_PROCESSOR_TIMEOUT")
    
    # Observability
    enable_metrics: bool = Field(default=True, env="ENABLE_METRICS")
    enable_tracing: bool = Field(default=False, env="ENABLE_TRACING")
    otel_exporter_otlp_endpoint: str = Field(
        default="http://localhost:4318",
        env="OTEL_EXPORTER_OTLP_ENDPOINT"
    )
    prometheus_port: int = Field(default=9090, env="PROMETHEUS_PORT")
    
    # Cost Tracking
    track_costs: bool = Field(default=True, env="TRACK_COSTS")
    cost_alert_threshold: float = Field(default=100.0, env="COST_ALERT_THRESHOLD")
    cost_currency: str = Field(default="USD", env="COST_CURRENCY")
    
    # PII Protection
    pii_redaction_enabled: bool = Field(default=True, env="PII_REDACTION_ENABLED")
    pii_patterns: List[str] = Field(
        default=["email", "phone", "ssn", "credit_card"],
        env="PII_PATTERNS"
    )
    
    # Rate Limiting
    rate_limit_enabled: bool = Field(default=True, env="RATE_LIMIT_ENABLED")
    rate_limit_per_minute: int = Field(default=60, env="RATE_LIMIT_PER_MINUTE")
    rate_limit_per_hour: int = Field(default=1000, env="RATE_LIMIT_PER_HOUR")
    
    # Testing
    test_mode: bool = Field(default=False, env="TEST_MODE")
    mock_providers: bool = Field(default=False, env="MOCK_PROVIDERS")
    
    @validator("allowed_origins", pre=True)
    def parse_allowed_origins(cls, v):
        """Parse comma-separated origins"""
        if isinstance(v, str):
            return [origin.strip() for origin in v.split(",")]
        return v
    
    @validator("pii_patterns", pre=True)
    def parse_pii_patterns(cls, v):
        """Parse comma-separated PII patterns"""
        if isinstance(v, str):
            return [pattern.strip() for pattern in v.split(",")]
        return v
    
    class Config:
        env_file = ".env"
        env_file_encoding = "utf-8"
        case_sensitive = False


# Global settings instance
settings = Settings()