"""
API endpoint tests
"""
import pytest
from httpx import AsyncClient


@pytest.mark.asyncio
async def test_root_endpoint(client: AsyncClient):
    """Test root endpoint"""
    response = await client.get("/")
    assert response.status_code == 200
    data = response.json()
    assert data["service"] == "AI Microservice"
    assert data["status"] == "running"


@pytest.mark.asyncio
async def test_health_check(client: AsyncClient):
    """Test health check endpoint"""
    response = await client.get("/health")
    assert response.status_code == 200
    data = response.json()
    assert "status" in data
    assert "services" in data


@pytest.mark.asyncio
async def test_create_ai_job(client: AsyncClient, api_headers):
    """Test creating AI job"""
    payload = {
        "job_type": "content_ideas",
        "tenant_id": "test-tenant",
        "owner_id": "test-user",
        "payload": {
            "topic": "AI in marketing",
            "count": 5
        }
    }
    
    response = await client.post(
        "/v1/ai/jobs",
        json=payload,
        headers=api_headers
    )
    
    assert response.status_code == 200
    data = response.json()
    assert "ai_job_id" in data
    assert data["status"] == "pending"


@pytest.mark.asyncio
async def test_create_ai_job_idempotency(client: AsyncClient, api_headers):
    """Test AI job idempotency"""
    payload = {
        "job_type": "content_ideas",
        "tenant_id": "test-tenant",
        "owner_id": "test-user",
        "payload": {"topic": "test"},
        "idempotency_key": "test-key-123"
    }
    
    # First request
    response1 = await client.post(
        "/v1/ai/jobs",
        json=payload,
        headers=api_headers
    )
    assert response1.status_code == 200
    job_id_1 = response1.json()["ai_job_id"]
    
    # Second request with same idempotency key
    response2 = await client.post(
        "/v1/ai/jobs",
        json=payload,
        headers=api_headers
    )
    assert response2.status_code == 200
    job_id_2 = response2.json()["ai_job_id"]
    
    # Should return same job
    assert job_id_1 == job_id_2


@pytest.mark.asyncio
async def test_get_job_status(client: AsyncClient, api_headers, sample_ai_job):
    """Test getting job status"""
    response = await client.get(
        f"/v1/ai/jobs/{sample_ai_job.id}",
        headers=api_headers
    )
    
    assert response.status_code == 200
    data = response.json()
    assert data["ai_job_id"] == sample_ai_job.id
    assert data["job_type"] == sample_ai_job.job_type


@pytest.mark.asyncio
async def test_create_publish_job(client: AsyncClient, api_headers):
    """Test creating publish job"""
    payload = {
        "platform": "twitter",
        "tenant_id": "test-tenant",
        "owner_id": "test-user",
        "content": {
            "text": "Test tweet"
        }
    }
    
    response = await client.post(
        "/v1/publish",
        json=payload,
        headers=api_headers
    )
    
    assert response.status_code == 200
    data = response.json()
    assert "publish_job_id" in data
    assert data["platform"] == "twitter"


@pytest.mark.asyncio
async def test_unauthorized_access(client: AsyncClient):
    """Test unauthorized access"""
    response = await client.post(
        "/v1/ai/jobs",
        json={"job_type": "test"}
    )
    
    assert response.status_code == 401


@pytest.mark.asyncio
async def test_metrics_endpoint(client: AsyncClient, admin_headers):
    """Test metrics endpoint"""
    response = await client.get(
        "/admin/metrics",
        headers=admin_headers
    )
    
    assert response.status_code == 200
    data = response.json()
    assert "total_jobs" in data
    assert "jobs_by_status" in data